# ESP8266 Light Control - Testing Guide

This project includes comprehensive automated unit tests using CMake build system.

## Quick Start - Running Tests

```bash
# Build and run all tests with CMake
mkdir build && cd build
cmake ..
make
ctest --output-on-failure

# Or run tests individually
./test_main_logic
./test_utils
./test_mqtt
./test_sensors
```

## Test Architecture

### CMake-Based Build

The project uses CMake for building tests, which provides:
- Platform-independent build configuration
- Automatic dependency management
- Integration with CTest for test execution
- Easy IDE integration

### Original Source Files

**Important:** Tests use the original `.ino` source files directly - **no code duplication**. The test framework:
1. Compiles original source files from `lamp_ctrl2_esp8266/`
2. Links them with test files
3. Test files only contain test cases and global variable definitions

## Test Coverage Summary

✅ **66 automated unit tests** covering:

### Main Logic Tests (19 tests)
- Pin state management (`set_out`, `set_out_pwm`)
- Power control (`get_power`, `set_power`, `switch_power`)
- Brightness control (`set_out_prc`)
- Color temperature control (`set_out_cct`)
- Edge cases and boundary validation

**Source**: `lamp_ctrl2_esp8266/main_logic.ino`  
**Tests**: `test/test_main_logic/test_main_logic.cpp`

### Utility Function Tests (14 tests)  
- MQTT status publishing (`send_status`, `send_io`, `send_power`)
- Device configuration (`send_config`, `send_device_info`)
- Sensor configuration (`send_sensor_config`)
- Brightness and CCT reporting (`send_prc`, `send_cct`)
- State synchronization (`send_all_state`)

**Source**: `lamp_ctrl2_esp8266/utils.ino`  
**Tests**: `test/test_utils/test_utils.cpp`

### MQTT Handler Tests (18 tests)
- Status message handling
- I/O command processing
- CCT control (Kelvin and mired)
- Brightness control (0-100%)
- Power commands
- Invalid input handling

**Source**: `lamp_ctrl2_esp8266/mqtt_utils.ino`  
**Tests**: `test/test_mqtt/test_mqtt.cpp`

### Sensor Tests (15 tests)
- Sensor initialization (`init_all_sensors`)
- Sensor lookup (`get_sensor_index`)
- Value management and isolation
- Metadata validation

**Source**: `lamp_ctrl2_esp8266/sens_logic.ino`  
**Tests**: `test/test_sensors/test_sensors.cpp`

## Test Results

```
Main Logic:    19 Tests  0 Failures  ✓
Utils:         14 Tests  0 Failures  ✓
MQTT:          18 Tests  0 Failures  ✓
Sensors:       15 Tests  0 Failures  ✓
─────────────────────────────────────
TOTAL:         66 Tests  0 Failures  ✓
```

## Build System Details

### CMake Configuration

The `CMakeLists.txt` file defines:
- Mock libraries for Arduino/ESP8266 dependencies
- Unity test framework (auto-downloaded)
- Source compilation from original `.ino` files
- Test executables with proper linking

### File Structure

```
├── CMakeLists.txt           # Main CMake configuration
├── lamp_ctrl2_esp8266/      # Original source files (.ino)
│   ├── main_logic.ino       # I/O control functions
│   ├── utils.ino            # Utility functions
│   ├── mqtt_utils.ino       # MQTT handlers
│   └── sens_logic.ino       # Sensor functions
├── test/
│   ├── mocks/               # Mock Arduino/ESP8266 libraries
│   ├── lamp_sources.cpp     # Wrapper that includes .ino files
│   ├── lamp_sources.h       # Function declarations
│   ├── test_main_logic/     # Main logic tests
│   ├── test_utils/          # Utility function tests
│   ├── test_mqtt/           # MQTT handler tests
│   └── test_sensors/        # Sensor function tests
└── build/                   # CMake build directory
```

### How It Works

1. **Source Wrapper** (`test/lamp_sources.cpp`):
   - Includes original `.ino` files as C++ code
   - Declares global variables as `extern`
   - Provides mock implementations for Arduino functions

2. **Test Files**:
   - Define global variables needed by source code
   - Include `lamp_sources.h` for function declarations
   - Contain only test cases - no function duplicates

3. **CMake Compilation**:
   - Compiles `lamp_sources.cpp` as object library
   - Compiles each test file
   - Links test files with source objects and mocks

## Building the Firmware

The project uses **arduino-cli** for building firmware (as shown in `.gitlab-ci.yml`):

```bash
arduino-cli compile --fqbn esp8266:esp8266:d1_mini_clone lamp_ctrl2_esp8266/lamp_ctrl2_esp8266.ino
```

## CI/CD Integration

Tests can be integrated into GitLab CI/CD pipeline:

```yaml
test:
  stage: test
  script:
    - mkdir build && cd build
    - cmake ..
    - make
    - ctest --output-on-failure
  tags:
    - shell
```

## Development Workflow

1. **Modify Source** - Edit files in `lamp_ctrl2_esp8266/`
2. **Build Tests** - `cd build && make`
3. **Run Tests** - `ctest` or `./test_main_logic`
4. **Verify** - All tests should pass
5. **Build Firmware** - Use arduino-cli to compile

## Requirements

### For Testing
- CMake 3.10 or later
- g++ compiler with C++11 support
- Internet connection (for Unity download on first build)

### For Building Firmware  
- arduino-cli
- ESP8266/ESP32 cores and libraries (see `.gitlab-ci.yml`)

## Benefits

✓ **No Code Duplication** - Tests use original source files  
✓ **CMake Build System** - Modern, platform-independent  
✓ **Fast Execution** - < 1 second for full test suite  
✓ **Easy Maintenance** - Changes to source automatically reflected in tests  
✓ **CI/CD Ready** - Automated testing in build pipeline

## More Information

- [Test Framework Details](test/README.md)
- [GitLab CI Configuration](.gitlab-ci.yml)
- [CMake Configuration](CMakeLists.txt)
