// IR Remote Control Support
// This file handles IR receiver functionality for controlling the lamp

#ifdef IR_PIN

#include <IRrecv.h>
#include <IRutils.h>

// IR receiver object
IRrecv irrecv(IR_PIN);
decode_results ir_results;

// NEC repeat code constant
#define IR_NEC_REPEAT_CODE 0xFFFFFFFF

// Store last command for repeat handling
uint16_t ir_last_addr = 0;
uint16_t ir_last_cmd = 0;

// Brightness step for increment/decrement (in percent)
#ifndef IR_BRIGHTNESS_STEP
#define IR_BRIGHTNESS_STEP 10
#endif

// CCT step for increment/decrement (in Kelvin)
#ifdef CCT_CONTROL
#ifndef IR_CCT_STEP
#define IR_CCT_STEP 300
#endif

// Get current CCT in Kelvin
unsigned long ir_get_current_cct() {
    return map(main_state.led_cct, 0, PWM_MAX, CCT_START, CCT_END);
}
#endif

// Initialize IR receiver
void ir_init() {
    irrecv.enableIRIn();
    debug_printf("IR receiver initialized on pin %d", IR_PIN);
}

// Get current brightness as percentage (0-100)
int ir_get_brightness_prc() {
#ifdef PWM_CONTROL
    return map(main_state.on_procents, 0, PWM_MAX, 0, 100);
#else
    return map(main_state.on_procents, 0, PinsCount, 0, 100);
#endif
}

#ifdef IR_CODE_TYPE_NEC
// Check if received NEC code matches expected address and command
// Uses the decoded address and command from IRremoteESP8266
bool ir_match_code(uint16_t received_addr, uint16_t received_cmd, uint8_t expected_addr, uint8_t expected_cmd) {
    // Compare lower 8 bits of address and command
    bool match = ((received_addr & 0xFF) == expected_addr) && ((received_cmd & 0xFF) == expected_cmd);
    
    debug_printf("IR match: recv A=0x%04X C=0x%04X, expect A=0x%02X C=0x%02X, match=%d", 
                 received_addr, received_cmd, expected_addr, expected_cmd, match);
    
    return match;
}

// Check if the last command is one that supports repeat (hold) - brightness +/- and CCT +/-
bool ir_is_repeat_supported_cmd(uint16_t addr, uint16_t cmd) {
#ifdef IR_CODE_BRI_PLUS
    {
        const uint8_t _code[] = IR_CODE_BRI_PLUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) return true;
    }
#endif
#ifdef IR_CODE_BRI_MINUS
    {
        const uint8_t _code[] = IR_CODE_BRI_MINUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) return true;
    }
#endif
#ifdef CCT_CONTROL
#ifdef IR_CODE_COLOR_TEMP_PLUS
    {
        const uint8_t _code[] = IR_CODE_COLOR_TEMP_PLUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) return true;
    }
#endif
#ifdef IR_CODE_COLOR_TEMP_MINUS
    {
        const uint8_t _code[] = IR_CODE_COLOR_TEMP_MINUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) return true;
    }
#endif
#endif
    return false;
}
#endif

// Process received IR command
void ir_process_command(uint16_t addr, uint16_t cmd) {
#ifdef IR_CODE_TYPE_NEC
    debug_printf("Processing IR: addr=0x%04X cmd=0x%04X", addr, cmd);
    
#ifdef IR_CODE_POWER
    {
        const uint8_t _code[] = IR_CODE_POWER;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Power toggle");
            set_power(!get_power());
            return;
        }
    }
#endif
    
#ifdef IR_CODE_BRI_PLUS
    {
        const uint8_t _code[] = IR_CODE_BRI_PLUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Brightness +");
            int current_prc = ir_get_brightness_prc();
            int new_prc = min(current_prc + IR_BRIGHTNESS_STEP, 100);
            if (!get_power()) set_power(true);
            set_out_prc(new_prc);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_BRI_MINUS
    {
        const uint8_t _code[] = IR_CODE_BRI_MINUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Brightness -");
            int current_prc = ir_get_brightness_prc();
            int new_prc = max(current_prc - IR_BRIGHTNESS_STEP, 1);
            set_out_prc(new_prc);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_BRI_20
    {
        const uint8_t _code[] = IR_CODE_BRI_20;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Brightness 20%%");
            if (!get_power()) set_power(true);
            set_out_prc(20);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_BRI_50
    {
        const uint8_t _code[] = IR_CODE_BRI_50;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Brightness 50%%");
            if (!get_power()) set_power(true);
            set_out_prc(50);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_BRI_100
    {
        const uint8_t _code[] = IR_CODE_BRI_100;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Brightness 100%%");
            if (!get_power()) set_power(true);
            set_out_prc(100);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_NIGHT_MODE
    {
        const uint8_t _code[] = IR_CODE_NIGHT_MODE;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Night mode");
            if (!get_power()) set_power(true);
            set_out_prc(1);
            return;
        }
    }
#endif
    
#ifdef CCT_CONTROL
#ifdef IR_CODE_COLOR_TEMP_MINUS
    {
        const uint8_t _code[] = IR_CODE_COLOR_TEMP_MINUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Color temp -");
            unsigned long current_cct = ir_get_current_cct();
            unsigned long new_cct = max((long)current_cct - IR_CCT_STEP, (long)CCT_START);
            set_out_cct(new_cct);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_COLOR_TEMP_PLUS
    {
        const uint8_t _code[] = IR_CODE_COLOR_TEMP_PLUS;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Color temp +");
            unsigned long current_cct = ir_get_current_cct();
            unsigned long new_cct = min(current_cct + IR_CCT_STEP, (unsigned long)CCT_END);
            set_out_cct(new_cct);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_COLOR_TEMP_WARM
    {
        const uint8_t _code[] = IR_CODE_COLOR_TEMP_WARM;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Color temp warm");
            set_out_cct(CCT_START);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_COLOR_TEMP_COLD
    {
        const uint8_t _code[] = IR_CODE_COLOR_TEMP_COLD;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Color temp cold");
            set_out_cct(CCT_END);
            return;
        }
    }
#endif
    
#ifdef IR_CODE_COLOR_TEMP_WHITE
    {
        const uint8_t _code[] = IR_CODE_COLOR_TEMP_WHITE;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Color temp neutral");
            unsigned long mid_cct = (CCT_START + CCT_END) / 2;
            set_out_cct(mid_cct);
            return;
        }
    }
#endif
#endif // CCT_CONTROL

#ifdef IR_CODE_AUTO
    {
        const uint8_t _code[] = IR_CODE_AUTO;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Auto mode");
            int current_prc = ir_get_brightness_prc();
            if (!get_power()) {
                set_power(true);
                set_out_prc(20);
            } else if (current_prc <= 20) {
                set_out_prc(50);
            } else if (current_prc <= 50) {
                set_out_prc(100);
            } else {
                set_out_prc(20);
            }
            return;
        }
    }
#endif
    
#ifdef IR_CODE_ROUND
    {
        const uint8_t _code[] = IR_CODE_ROUND;
        if (ir_match_code(addr, cmd, _code[0], _code[1])) {
            debug_printf("IR: Round/demo mode");
            // Could implement a demo sequence here
            // For now, just toggle power
            set_power(!get_power());
            return;
        }
    }
#endif
    
    debug_printf("IR: Unknown code");
#endif // IR_CODE_TYPE_NEC
}

// Process IR receiver - call this in main loop
void ir_loop() {
    if (irrecv.decode(&ir_results)) {
#ifdef IR_CODE_TYPE_NEC
        if (ir_results.decode_type == NEC) {
            debug_printf("IR NEC received: value=0x%08X address=0x%04X command=0x%04X", 
                         (uint32_t)ir_results.value, ir_results.address, ir_results.command);
            
            // Handle repeat codes (0xFFFFFFFF for NEC) - only for +/- buttons
            if ((uint32_t)ir_results.value == IR_NEC_REPEAT_CODE) {
                // On repeat, re-execute last command only if it's a +/- button
                if (ir_is_repeat_supported_cmd(ir_last_addr, ir_last_cmd)) {
                    debug_printf("IR repeat: re-executing last +/- command");
                    ir_process_command(ir_last_addr, ir_last_cmd);
                }
            } else {
                // New command - save and execute
                ir_last_addr = ir_results.address;
                ir_last_cmd = ir_results.command;
                ir_process_command(ir_results.address, ir_results.command);
            }
        }
#endif
        irrecv.resume();
    }
}

#endif // IR_PIN
