#ifdef ESP8266
#include <ESP8266WiFi.h>
#include <ESP8266mDNS.h>
#include <ESP8266HTTPClient.h>
#include <ESP8266httpUpdate.h>
#include <ESP8266WebServer.h>
#define WIFI_STATUS WiFi.status()
#define HTTPUPDATER ESPhttpUpdate
#elif defined(ESP32)
#include <WiFi.h>
#include <WiFiMulti.h>
#include <WebServer.h>
#include <HTTPClient.h>
#include <HTTPUpdate.h>
#include <esp_mac.h>
#define WIFI_STATUS WiFiMulti.run()
#define HTTPUPDATER httpUpdate
#endif

#include <PubSubClient.h>
#include <WiFiUdp.h>
#include <ArduinoJson.h>
#include <FileData.h>
#include <LittleFS.h>
#include <Ticker.h>
#define DEBUG

#ifndef EXT_CONFIG
//#define NO_UPDATE
//#include "configs/config_lamp.h"
#include "configs/config_rgblamp.h"
//#include "configs/config_led_lamp.h"
//#include "configs/config_led_strip.h"
//#include "configs/config_led_backlight_esp32.h"
//#include "configs/config_2_outlet.h"
#endif

const size_t PinsCount = sizeof(pin_state) / sizeof(pin_state[0]);

#ifdef INA_SENS
#include <GyverINA.h>
#endif

#ifdef WS_PIN
#include <Adafruit_NeoPixel.h>
#endif

#ifdef SHARP_IR_GP2Y0A21YK0F
#include <ESP32SharpIR.h>

ESP32SharpIR proxy_sensor( ESP32SharpIR::GP2Y0A21YK0F, SHARP_PIN);
#endif

#ifdef IR_PIN
#include <IRremoteESP8266.h>
#endif

#ifdef WS_PIN
Adafruit_NeoPixel ws_leds(WS_NUM, WS_PIN, NEO_GRB + NEO_KHZ800);
#endif

#ifdef DEBUG
#define ENABLE_NET_DEBUG
#endif

#ifdef ENABLE_NET_DEBUG
IPAddress m_ip(239, 243, 42, 19);
unsigned int m_port = 6219;
#endif

#define BUF_SIZE 2048 //Размер буффера для формирование страниц
#define MSG_BUFFER_SIZE    (50)
#define STATUS_SEND_INT 120
#define SENSORS_SEND_INT 10
#define RECONNECT_INTERVAL 10 * 1000 //30 sec
#define RECONNECT_MQTT_INTERVAL 3 * 1000 //30 sec
#define SECS_PER_MIN  (60UL) //Кол секунд в минуту
#define SECS_PER_HOUR (3600UL) //Кол секунд в час
#define SECS_PER_DAY  (SECS_PER_HOUR * 24L) //Кол секунд в день
#define numberOfSeconds(_time_) (_time_ % SECS_PER_MIN)  //Перевод микросекунд в секунды
#define numberOfMinutes(_time_) ((_time_ / SECS_PER_MIN) % SECS_PER_MIN) //Перевод микросекунд в минуты
#define numberOfHours(_time_) (( _time_% SECS_PER_DAY) / SECS_PER_HOUR) //Перевод микросекунд в часы
#define elapsedDays(_time_) ( _time_ / SECS_PER_DAY) //Перевод микросекунд в дни

#include "base/vendor_config.h"

#ifdef ESP32
WiFiMulti WiFiMulti;
#endif

WiFiUDP udp;
WiFiClient espClient;
PubSubClient client(espClient);
#ifdef ESP8266
#ifndef POWER_DETECT_ESP8266_ADC
ADC_MODE(ADC_VCC);
#endif
ESP8266WebServer httpServer(80); // Указываем порт Web-сервера
#elif defined(ESP32)
WebServer httpServer(80);
const int pwm_res = (int)(log(PWM_MAX+1)/log(2));
#endif
Ticker status_send_tic; //Обьявление таймеров событий
#ifdef INA_SENS
#ifdef INA_SHUT
INA219 ina(INA_SHUT, INA_MAX, INA_ADDR);
#else
INA219 ina;
#endif

bool ina219_init_ok = false;
#endif

Ticker sensors_send_tic; //Обьявление таймеров событий

volatile bool send_status_flag = false;
volatile bool send_sensor_flag = false;
volatile bool power_pressed_flag = false;

unsigned long power_pressed_timer = 0;

char cstr1[BUF_SIZE];
unsigned long previousMillis = 0;
char msg[MSG_BUFFER_SIZE];

String device_topic;
String device_sensor_topic;
String cmd_topic;
String state_topic;
String announcement_msg = "";
String clientId;

#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
struct main_state_str main_state = {
        .on_procents = 0,
#ifdef CCT_CONTROL
        .led_cct = 0,
#endif
#ifdef WSRGB
        .rgb_r = 255,
        .rgb_g = 255,
        .rgb_b = 255
#endif
};
#endif

FileData pinStatesFile(&LittleFS, "/pinStates.dat", 'B', &pin_state, sizeof(pin_state));
#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
FileData mainStateFile(&LittleFS, "/mainState.dat", 'B', &main_state, sizeof(main_state));
#endif

#ifdef WSRGB
// Custom WS2813 bit-banging function - sends data using delays without disabling interrupts
// This allows setting LED colors before WiFi init without breaking OTA updates
void ws2813_send_byte(uint8_t byte) {
    // Use direct GPIO manipulation for faster timing
    // On ESP8266, GPOS sets pin HIGH, GPOC sets pin LOW
    // This is much faster than digitalWrite (~100ns vs ~3-4us)
    uint32_t pin_mask = 1 << WS_PIN;
    
    for(int8_t i = 7; i >= 0; i--) {
        if(byte & (1 << i)) {
            // Send '1': ~0.7-0.8us HIGH, ~0.6us LOW
            GPOS = pin_mask;  // Set HIGH
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            GPOC = pin_mask;  // Set LOW
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
        } else {
            // Send '0': ~0.35us HIGH, ~0.8us LOW
            GPOS = pin_mask;  // Set HIGH
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            GPOC = pin_mask;  // Set LOW
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
            __asm__ __volatile__("nop; nop; nop; nop; nop; nop; nop; nop;");
        }
    }
}

void ws2813_show_early(uint8_t r, uint8_t g, uint8_t b, uint8_t num_leds) {
    pinMode(WS_PIN, OUTPUT);
    digitalWrite(WS_PIN, LOW);
    delayMicroseconds(60);  // Reset pulse
    
    // Send GRB data for each LED
    for(uint8_t i = 0; i < num_leds; i++) {
        ws2813_send_byte(g);
        ws2813_send_byte(r);
        ws2813_send_byte(b);
    }
    
    digitalWrite(WS_PIN, LOW);
    delayMicroseconds(60);  // Latch
}
#endif

void sendHeaders() {
    httpServer.sendHeader("Server", "ESP_SD/0.0.1", true);
    httpServer.sendHeader("X-Content-Type-Options", "nosniff", false);
    httpServer.sendHeader("Content-Language", "ru", false);
    httpServer.sendHeader("Retry-After", "20", false);
    return;
}

void update_started() {
    Serial.println("CALLBACK:  HTTP update process started");
}

void update_progress(int cur, int total) {
    yield();
#ifdef ESP8266
    ESP.wdtFeed();
#endif
    Serial.printf("CALLBACK:  HTTP update process at %d of %d bytes...\n", cur, total);
}

void setup_wifi() {

    delay(10);
    // We start by connecting to a WiFi network
    Serial.println();
    debug_printf("Connecting to %s", ssid);

#ifdef ESP8266
    WiFi.mode(WIFI_STA);
    WiFi.hostname(dev_name);
    WiFi.begin(ssid, password);
#elif defined(ESP32)
    WiFi.setHostname(dev_name);
    WiFi.mode(WIFI_STA);
    WiFiMulti.addAP(ssid, password);
#endif

    for (uint8_t t = 0; WIFI_STATUS != WL_CONNECTED && t < 30; t++) {
        delay(500);
        yield();
        Serial.print(".");
    }

    Serial.println("");
    debug_printf("WiFi connected ");
    debug_printf("IP address: %s", WiFi.localIP().toString().c_str());
#ifdef ESP8266
    MDNS.begin(dev_name);
#endif
    yield();
#ifndef NO_UPDATE
    process_update();
#endif
}

void process_update() {

    if (WIFI_STATUS != WL_CONNECTED)
        return;

    HTTPUPDATER.onStart(update_started);
    HTTPUPDATER.onProgress(update_progress);

    debug_printf("Starting fw updating");
    client.publish(device_topic.c_str(), "Starting fw updating");
    HTTPUPDATER.rebootOnUpdate(false);
    t_httpUpdate_return ret = HTTPUPDATER.update(espClient, String(update_endpoint) + String(update_file));

    switch (ret) {
        case HTTP_UPDATE_FAILED:
            client.publish(device_topic.c_str(), "HTTP_UPDATE_FAILD");
            debug_printf("HTTP_UPDATE_FAILD Error (%d): %s\n", HTTPUPDATER.getLastError(),
                         HTTPUPDATER.getLastErrorString().c_str());
            break;

        case HTTP_UPDATE_NO_UPDATES:
            debug_printf("HTTP_UPDATE_NO_UPDATES");
            client.publish(device_topic.c_str(), "HTTP_UPDATE_NO_UPDATES");
            break;

        case HTTP_UPDATE_OK:
            debug_printf("HTTP_UPDATE_OK");
            client.publish(device_topic.c_str(), "HTTP_UPDATE_OK");
            set_out(0, OFF);
            delay(500);
            set_out(0, ON);
            delay(500);
            set_out(0, OFF);
            delay(500);
            set_out(0, ON);
            ESP.restart(); //В случае успешной перепрошивки перезапуск
            break;
    }
}

String get_sta_mac() {
    uint8_t mac[6];
    char macStr[14] = {0};
#ifdef ESP8266
    wifi_get_macaddr(STATION_IF, mac);
#elif defined(ESP32)
    esp_base_mac_addr_get(mac);
#endif

    sprintf(macStr, "%02X%02X%02X%02X%02X%02X", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
    return String(macStr);
}

void setup() {
    Serial.begin(115200);
    //Setup pwm settings
#ifdef PWM_CONTROL
#ifdef ESP8266
    analogWriteRange(PWM_MAX);
    analogWriteFreq(PWM_FRQ);
#elif defined(ESP32)
    for(size_t ledChannel = 0; ledChannel < PinsCount; ledChannel++) {
      //ledcSetup(ledChannel, PWM_FRQ, pwm_res);
      ledcAttach(pin_state[ledChannel].pin, PWM_FRQ, pwm_res);
      //debug_printf("ledcSetup(%d, %d, %d)", ledChannel, PWM_FRQ, pwm_res);
      //ledcAttachPin(pin_state[ledChannel].pin, ledChannel);
      //debug_printf("ledcAttachPin(%d, %d)", pin_state[ledChannel].pin, ledChannel);
    }
#endif
#endif

    if(!LittleFS.begin()) {
      debug_printf("Fail to init LittleFS");
      LittleFS.format();
      ESP.restart();
      }

    FDstat_t stat = pinStatesFile.read();

#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
    FDstat_t main_stat = mainStateFile.read();
#endif

    //INIT IO
    for (size_t i = 0; i < PinsCount; i++) {
#ifdef PWM_CONTROL
#ifdef ESP8266
        pinMode(pin_state[i].pin, OUTPUT);
        if (pin_state[i].state == ON)
            analogWrite(pin_state[i].pin, pin_state[i].pwm);
        else
            analogWrite(pin_state[i].pin, OFF);
#elif defined(ESP32)
        if (pin_state[i].state == ON)
            ledcWrite(pin_state[i].pin, pin_state[i].pwm);
        else
            ledcWrite(pin_state[i].pin, OFF);
#endif
#else
        pinMode(pin_state[i].pin, OUTPUT);
#ifdef PULSE_CONTROL
        pinMode(pin_state[i].state_pin, INPUT);
#endif
        digitalWrite(pin_state[i].pin, pin_state[i].state);
#endif
    }

    debug_printf("Readed data:");
    for (size_t i = 0; i < PinsCount; i++) {
#ifdef PWM_CONTROL
        debug_printf("pin_state[%d]: pin=%d, state=%d, pwm=%d", i, pin_state[i].pin, pin_state[i].state,
                     pin_state[i].pwm);
#else
        debug_printf("pin_state[%d]: pin=%d, state=%d", i, pin_state[i].pin, pin_state[i].state);
#endif
    }

#ifdef WSRGB
    // Early LED initialization using custom bit-banging (no library, no interrupt disable)
    // Apply saved color with brightness
    uint8_t early_brightness = main_state.on_procents;
    uint8_t early_r = (main_state.rgb_r * early_brightness) / 255;
    uint8_t early_g = (main_state.rgb_g * early_brightness) / 255;
    uint8_t early_b = (main_state.rgb_b * early_brightness) / 255;
    
    ws2813_show_early(early_r, early_g, early_b, WS_NUM);
    debug_printf("WS LEDs set early with custom function: R=%d G=%d B=%d", early_r, early_g, early_b);
#endif

    init_all_sensors();
    setup_wifi();

    debug_printf("Readed data:");
    for (size_t i = 0; i < PinsCount; i++) {
#ifdef PWM_CONTROL
        debug_printf("pin_state[%d]: pin=%d, state=%d, pwm=%d", i, pin_state[i].pin, pin_state[i].state,
                     pin_state[i].pwm);
#else
        debug_printf("pin_state[%d]: pin=%d, state=%d", i, pin_state[i].pin, pin_state[i].state);
#endif
    }

    debug_printf("Restored config data state:");
    switch (stat) {
        case FD_FS_ERR:
            debug_printf("FS Error");
            break;
        case FD_FILE_ERR:
            debug_printf("Error");
            break;
        case FD_WRITE:
            debug_printf("Data Write");
            break;
        case FD_ADD:
            debug_printf("Data Add");
            break;
        case FD_READ:
            debug_printf("Data Read");
            break;
        default:
            break;
    }

    device_topic = String(topic_prefix) + "/" + String(device_type) + "/" + String(endpoint_type) + "_" + get_sta_mac();
    device_sensor_topic = String(topic_prefix) + "/sensor/" + String(endpoint_type) + "_" + get_sta_mac();
    cmd_topic = device_topic + "/cmd";
    state_topic = device_topic + "/state";
    clientId = "ESP8266-" + String(device_type) + "-" + get_sta_mac();
    announcement_msg = "My ip is " + WiFi.localIP().toString();

    debug_printf("Starting mqtt");
    debug_printf("device_topic: %s", device_topic.c_str());
    debug_printf("cmd_topic: %s", cmd_topic.c_str());
    debug_printf("state_topic: %s", state_topic.c_str());
    debug_printf("clientId: %s", clientId.c_str());
    debug_printf("announcement_msg: %s", announcement_msg.c_str());
    httpServer.on("/", []() {
        bzero(cstr1, BUF_SIZE);
        sendHeaders();
#ifdef DEBUG
        httpServer.sendHeader("Refresh", "10", false);
        /*Вывод имя хоста и версии прошивки*/
        sprintf_P(cstr1, PSTR("Hostname: %s\n"), dev_name);
        sprintf_P(cstr1, PSTR("Software ver: %s\n"), vendor_info.sw_version);
        /*Вывод информации о компиляции*/
        sprintf_P(cstr1, PSTR("%sComp info, time: " __TIME__ ", date: " __DATE__ "\n"), cstr1);
#ifdef BUILDINFO
        sprintf_P(cstr1, PSTR("%sGitLab build info: " BUILDINFO ", SHA: " BUILD_SHA "\n"), cstr1);
#endif
        /*Вывод мак адреса*/
        sprintf_P(cstr1, PSTR("%sMAC: %s\n"), cstr1, get_sta_mac().c_str());
        sprintf_P(cstr1, PSTR("%sSketchSize: %d\n"), cstr1, ESP.getSketchSize()); //Размер скетча
        sprintf_P(cstr1, PSTR("%sSketchMD5: %s\n"), cstr1, ESP.getSketchMD5().c_str()); //Хеш скетча

        sprintf_P(cstr1, PSTR("%sCPU Frq: %d MHz\n"), cstr1, ESP.getCpuFreqMHz()); //Частота цп
        sprintf_P(cstr1, PSTR("%sFree Heap: %d\n"), cstr1, ESP.getFreeHeap()); //Свободное ОЗУ
#if defined(ESP8266) && !defined(POWER_DETECT_ESP8266_ADC)
        sprintf_P(cstr1, PSTR("%sESP VCC: %.2f\n"), cstr1, ESP.getVcc() / 1000.0); //Наприжение питания
#endif
        unsigned long tfs = millis() / 1000; //Получение времени с момента старта
        sprintf_P(cstr1, PSTR("%sTime from start: %lu hours %lu minutes %lu Sec, and %lu days \n"), cstr1, numberOfHours(tfs), numberOfMinutes(tfs), numberOfSeconds(tfs), elapsedDays(tfs));
#ifdef ESP8266
        sprintf_P(cstr1, PSTR("%sESP Chip ID: %X\n"), cstr1, ESP.getChipId()); //ID чипа
        sprintf_P(cstr1, PSTR("%sFlash real id: %X\n"), cstr1, ESP.getFlashChipId()); //ID микрочипа флеш памяти
        sprintf_P(cstr1, PSTR("%sFlash real size: %d\n"), cstr1, ESP.getFlashChipRealSize()); //Размер памяти по ID
        sprintf_P(cstr1, PSTR("%sFlash ide size: %d\n"), cstr1, ESP.getFlashChipSize()); //Размер памяти по уст компилятора
        sprintf_P(cstr1, PSTR("%sFlash ide speed: %d\n"), cstr1, ESP.getFlashChipSpeed()); //Скорость памяти
        /*Режим работы памяти*/
        sprintf_P(cstr1, PSTR("%sFlash ide mode: %s\n"), cstr1, (ESP.getFlashChipMode() == FM_QIO ? "QIO" : ESP.getFlashChipMode() == FM_QOUT ? "QOUT" : ESP.getFlashChipMode() == FM_DIO ? "DIO" : ESP.getFlashChipMode() == FM_DOUT ? "DOUT" : "UNKNOWN"));
        sprintf_P(cstr1, PSTR("%sLast reset: %s\n"), cstr1, ESP.getResetReason().c_str()); //Причина последней перезагрузки
        sprintf_P(cstr1, PSTR("%sReset info: %s\n"), cstr1, ESP.getResetInfo().c_str()); //Описание
#endif
        sprintf_P(cstr1, PSTR("%sWiFi Status: %d\n"), cstr1, WIFI_STATUS); //Статус соеденения
        sprintf_P(cstr1, PSTR("%sSSID: %s\n"), cstr1, WiFi.SSID().c_str()); //Имя беспроводной сети
        sprintf_P(cstr1, PSTR("%sRSSI: %d dbm\n"), cstr1, WiFi.RSSI()); //Уровень сигнала в дБм

        sprintf_P(cstr1, PSTR("%sDebug data:\n"), cstr1);               // Веб-страница создается с использованием HTML

        sprintf_P(cstr1, PSTR("%s clientId=%s\n"), cstr1, clientId.c_str());
        sprintf_P(cstr1, PSTR("%s device_topic=%s\n"), cstr1, device_topic.c_str());
        sprintf_P(cstr1, PSTR("%s cmd_topic=%s\n"), cstr1, cmd_topic.c_str());
    //    sprintf_P(cstr1, PSTR("%s value=%d\n"), cstr1, value);
#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
        sprintf_P(cstr1, PSTR("%s on_procents=%u\n"), cstr1, main_state.on_procents);
#endif
#ifdef PWM_CONTROL
        sprintf_P(cstr1, PSTR("%s PWM_CONTROL ENABLED\n"), cstr1);
        sprintf_P(cstr1, PSTR("%s  PWM_MAX=%u\n"), cstr1, PWM_MAX);
        sprintf_P(cstr1, PSTR("%s  PWM_FRQ=%u\n"), cstr1, PWM_FRQ);
        //sprintf_P(cstr1, PSTR("%s  led_bri=%u\n"), cstr1, led_bri);
#endif
#ifdef CCT_CONTROL
        sprintf_P(cstr1, PSTR("%s CCT_CONTROL ENABLED\n"), cstr1);
        sprintf_P(cstr1, PSTR("%s  MIN_TEMP=%u\n  MAX_TEMP=%u\n"), cstr1, CCT_START, CCT_END);
        sprintf_P(cstr1, PSTR("%s  led_cct=%u\n"), cstr1, main_state.led_cct);
#endif
#ifdef WSRGB
        sprintf_P(cstr1, PSTR("%s WSRGB ENABLED\n"), cstr1);
        sprintf_P(cstr1, PSTR("%s  RGB: R=%u G=%u B=%u\n"), cstr1, main_state.rgb_r, main_state.rgb_g, main_state.rgb_b);
        sprintf_P(cstr1, PSTR("%s  Brightness=%u\n"), cstr1, main_state.on_procents);
#endif
        sprintf_P(cstr1, PSTR("%s ON=%d OFF=%d\n"), cstr1, ON, OFF);
        for (size_t i = 0; i < PinsCount; i++)
        {
          sprintf_P(cstr1, PSTR("%s  pin_state[%d].pin=%d\n"), cstr1, i, pin_state[i].pin);
          sprintf_P(cstr1, PSTR("%s  pin_state[%d].state=%d\n"), cstr1, i, pin_state[i].state);
#ifdef PWM_CONTROL
          sprintf_P(cstr1, PSTR("%s  pin_state[%d].pwm=%d\n"), cstr1, i, pin_state[i].pwm);
#endif
        }
#else
        sprintf_P(cstr1, PSTR("No data\n"));
#endif
        httpServer.send(200, "text/plain; charset=utf-8", cstr1);
    });
    httpServer.begin();
    client.setServer(mqtt_server, 1883);
    client.setCallback(callback);
#ifdef INA_SENS
    if (ina.begin()) {
        debug_printf("ina219 init success");
        ina219_init_ok = true;
        #ifdef INA_CAL
        ina.setCalibration(INA_CAL);
        #endif
        ina.setResolution(INA219_VBUS, INA219_RES_12BIT_X4);      // Напряжение в 12ти битном режиме + 4х кратное усреднение
        ina.setResolution(INA219_VSHUNT, INA219_RES_12BIT_X4);  // Ток в 12ти битном режиме + 128х кратное усреднение
    } else {
        debug_printf("ina219 init fail");
    }
#endif

#ifdef WS_PIN
    // Initialize WS LED library for normal control after update check
    // (LEDs were already lit using custom bit-banging function before WiFi init)
    ws_leds.begin();
    ws_leds.clear();
#ifdef WSRGB
    // Update with current saved state
    uint8_t lib_brightness = main_state.on_procents;
    for(int i=0; i<WS_NUM; i++) {
        uint8_t lib_r = (main_state.rgb_r * lib_brightness) / 255;
        uint8_t lib_g = (main_state.rgb_g * lib_brightness) / 255;
        uint8_t lib_b = (main_state.rgb_b * lib_brightness) / 255;
        ws_leds.setPixelColor(i, ws_leds.Color(lib_r, lib_g, lib_b));
    }
    ws_leds.show();
    debug_printf("WS LEDs library control restored");
#else
    for(int i=0; i<WS_NUM; i++) {
        ws_leds.setPixelColor(i, ws_leds.Color(127, 127, 127));
    }
    ws_leds.show();
#endif
#endif
    status_send_tic.attach(STATUS_SEND_INT, status_send_f);
    sensors_send_tic.attach(SENSORS_SEND_INT, sensors_send_f);
#ifdef IR_PIN
    ir_init();
#endif
}

void loop() {
    if (pinStatesFile.tick() == FD_WRITE)
        debug_printf("pin state data updated!");
#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
    if (mainStateFile.tick() == FD_WRITE)
        debug_printf("main state data updated!");
#endif
#ifdef ESP32
    //Autoreconnect for esp32
    yield();
    if ((WIFI_STATUS != WL_CONNECTED) && (millis() - previousMillis >= RECONNECT_INTERVAL)) {
        debug_printf("%l",millis());
        debug_printf("Reconnecting to WiFi...");
        WiFi.disconnect();
        WiFi.reconnect();
        previousMillis = millis();
    }
#endif
    if (!client.connected()) {
        reconnect();
    }
    client.loop();
    httpServer.handleClient();
#ifdef ESP8266
    ESP.wdtFeed();
#endif
    yield();
    if (send_status_flag == true) {
        send_status_flag = false;
        if (client.connected()) {
            send_all_state();
        }
    }
    yield();
    if (send_sensor_flag == true) {
        send_sensor_flag = false;
        processing_all_sensors(false);
        if (client.connected()) {
            send_all_sensors();
        }
    }
    processing_all_sensors(true);
    yield();
#ifdef IR_PIN
    ir_loop();
#endif

}
