size_t debug_printf(const char *format, ...) {
#ifdef DEBUG
#define PF_BUF_SIZE 1024
    char buf[PF_BUF_SIZE];
    snprintf(buf, PF_BUF_SIZE, "{\n  \"mac\":\"%s\",\n  \"host\":\"%s\",\n  \"text\":\"", WiFi.macAddress().c_str(), dev_name);
    size_t start_size = strlen(buf);
    va_list ap;
    va_start(ap, format);
    vsnprintf(buf + start_size, sizeof(buf) - ((start_size + 5)*sizeof(char)), format, ap);
    va_end(ap);
    size_t ret = Serial.write(buf + start_size);
    ret = ret + Serial.write("\n");
    snprintf(buf, PF_BUF_SIZE, "%s\"\n}\n", buf);

#ifdef ENABLE_NET_DEBUG
    if (WIFI_STATUS == WL_CONNECTED) {
      udp.beginPacket(m_ip, m_port + 1);
#ifdef ESP8266
      udp.write(buf);
#elif defined(ESP32)
      udp.print(buf);
#endif
      udp.endPacket();
    }
#endif
    return (ret);
#else
    return 0;
#endif
}

// Универсальная функция отправки состояния
void send_status_to_mqtt(const String &topic, const String &state) {
    if (client.connected()) {
        yield();
        send_status(topic, state);
    }
}

// Отправка состояния выводов
void send_io() {
    for (size_t i = 0; i < PinsCount; i++) {
        bool state = false;
        if(pin_state[i].state == ON)
            state = true;
        else
            state = false;
        send_status_to_mqtt(String(endpoint_type) + "/" + String(i), String(state));
    }
}

// Отправка состояния питания устройства
void send_power() {
    send_power(get_power());
}

void send_power(bool power_state) {
    send_status_to_mqtt(String(endpoint_type) + "/power", String(power_state));
}

// Отправка яркости
void send_prc(unsigned on_procents) {
    if (client.connected()) {
        yield();
#ifdef PWM_CONTROL
        unsigned real_procents = map(on_procents, 0, PWM_MAX, 0, 100);
#else
        unsigned real_procents = map(on_procents, 0, PinsCount, 0, 100);
#endif
        send_status(String(endpoint_type) + "/prc", String(real_procents));
    }
}

// Отправка цветовой температуры (CCT)
#ifdef CCT_CONTROL
void send_cct(unsigned led_cct) {
    if (client.connected()) {
        yield();
        unsigned long real_cct = map(led_cct, 0, PWM_MAX, CCT_START, CCT_END);
        send_status_to_mqtt(String(endpoint_type) + "/cct", String(real_cct));
        send_status_to_mqtt(String(endpoint_type) + "/cct_mired", String(1000000UL / real_cct));
        debug_printf("cct kelvin: %s mired: %s", String(real_cct).c_str(), String(1000000UL / real_cct));
    }
}
#endif

#ifdef WSRGB
// Отправка RGB цвета
void send_rgb_color() {
    if (client.connected()) {
        yield();
        JsonDocument color_doc;
        color_doc["r"] = main_state.rgb_r;
        color_doc["g"] = main_state.rgb_g;
        color_doc["b"] = main_state.rgb_b;
        String color_str;
        serializeJson(color_doc, color_str);
        send_status("rgb_color", color_str);
    }
}

// Отправка RGB яркости
void send_rgb_brightness() {
    if (client.connected()) {
        yield();
        send_status("rgb_brightness", String(main_state.on_procents));
    }
}

// Отправка RGB состояния питания
void send_rgb_power() {
    if (client.connected()) {
        yield();
        bool power_state = (main_state.on_procents > 0);
        send_status("rgb_power", String(power_state));
    }
}
#endif

// Отправка всех состояний
void send_all_state() {
    send_power();

#if defined(BRI_CONTROL) || defined(CCT_CONTROL)
    send_prc(main_state.on_procents);
#else
    send_io();
#endif
#ifdef CCT_CONTROL
    send_cct(main_state.led_cct);
#endif
#ifdef WSRGB
    send_rgb_power();
    send_rgb_brightness();
    send_rgb_color();
#endif
}

// Отправка конфигурации устройства
void send_config() {
    return send_config(0);
}

void send_config(unsigned conf_id) {
    if (!client.connected())
        return;

    JsonDocument config;
#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
    config["name"] = dev_name + String("_") + get_sta_mac();
    config["unique_id"] = endpoint_type + String("_") + get_sta_mac();
#else
    config["name"] = dev_name + String("_") + get_sta_mac() + "_" + String(conf_id);
    config["unique_id"] = endpoint_type + String("_") + get_sta_mac() + "_" + String(conf_id);
#endif
    config["schema"] = "basic";

    JsonArray supported_color_modes = config.createNestedArray("supported_color_modes");

    // Цветовые режимы
#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
    config["color_mode"] = true;
#else
    config["color_mode"] = false;
#endif

    // Настройка яркости
#if defined(BRI_CONTROL)
    config["brightness_command_topic"] = cmd_topic + "/" + String(endpoint_type);
    config["brightness_state_topic"] = state_topic + "/" + String(endpoint_type) + "/prc";
    config["brightness_scale"] = 100;
    config["brightness"] = true;
#elif defined(WSRGB)
    config["brightness_command_topic"] = cmd_topic + "/rgb_brightness";
    config["brightness_state_topic"] = state_topic + "/rgb_brightness";
    config["brightness_scale"] = 255;
    config["brightness"] = true;
#else
    config["brightness"] = false;
#endif

    // Добавление поддерживаемых режимов
#if defined(CCT_CONTROL)
    supported_color_modes.add("color_temp");
#elif defined(WSRGB)
    supported_color_modes.add("rgb");
#elif defined(BRI_CONTROL)
    supported_color_modes.add("brightness");
#else
    supported_color_modes.add("onoff");
#endif

    // Настройки CCT
#ifdef CCT_CONTROL
    config["color_temp"] = true;
    //config["min_color_temp_kelvin"]= CCT_START;
    //config["max_color_temp_kelvin"]= CCT_END;
    config["max_mireds"] = CCT_START_MIRED;
    config["min_mireds"] = CCT_END_MIRED;
    config["color_temp_command_topic"] = cmd_topic + "/cct_mired";
    //config["color_temp_state_topic"]=state_topic + "/" + String(endpoint_type) + "/cct";
    config["color_temp_state_topic"] = state_topic + "/" + String(endpoint_type) + "/cct_mired";
#else
    config["color_temp"] = false;
#endif

    // Настройки RGB
#ifdef WSRGB
    config["rgb_command_topic"] = cmd_topic + "/rgb_color";
    config["rgb_state_topic"] = state_topic + "/rgb_color";
    config["rgb_command_template"] = "{{ '{\"r\":' ~ red ~ ',\"g\":' ~ green ~ ',\"b\":' ~ blue ~ '}' }}";
    config["rgb_value_template"] = "{{ value_json.r }},{{ value_json.g }},{{ value_json.b }}";
#endif

    // Командные топики
#if defined(BRI_CONTROL) || defined(CCT_CONTROL)
    config["command_topic"] = cmd_topic + "/" + String(endpoint_type) + "_power";
    config["state_topic"] = state_topic + "/" + String(endpoint_type) + "/power";
#elif defined(WSRGB)
    config["command_topic"] = cmd_topic + "/rgb_power";
    config["state_topic"] = state_topic + "/rgb_power";
#else
    config["command_topic"] = cmd_topic + "/" + String(endpoint_type) + "/" + conf_id;
    config["state_topic"] = state_topic + "/" + String(endpoint_type) + "/" + conf_id;
#endif

    config["payload_on"] = 1;
    config["payload_off"] = 0;
    config["state_on"] = "1";
    config["state_off"] = "0";

    // Информация о устройстве
    send_device_info(config);

    // Отправка конфигурации
    String config_str;
    serializeJson(config, config_str);
    debug_printf("Size of config[%u]: %u, send config to mqtt", conf_id, config_str.length());
    client.setBufferSize(config_str.length() + 64);

#if defined(BRI_CONTROL) || defined(CCT_CONTROL) || defined(WSRGB)
    client.publish(String(device_topic + "/config").c_str(), config_str.c_str());
#else
    client.publish(String(device_topic + "_" + String(conf_id) + "/config").c_str(), config_str.c_str());
#endif
    client.setBufferSize(MQTT_MAX_PACKET_SIZE);

#if defined(WSRGB)
    // For WSRGB mode, also send config for relay pins if they exist
    if (conf_id == 0 && PinsCount > 0) {
        for (size_t i = 0; i < PinsCount; i++) {
            // Send config for each relay pin as a separate light entity
            // Format must match main branch: {prefix}_{MAC}_{index}
            JsonDocument relay_config;
            relay_config["name"] = String(dev_name) + "_" + get_sta_mac() + "_" + String(i);
            relay_config["unique_id"] = String(endpoint_type) + "_" + get_sta_mac() + "_" + String(i);
            relay_config["command_topic"] = cmd_topic + "/" + String(endpoint_type) + "/" + String(i);
            relay_config["state_topic"] = state_topic + "/" + String(endpoint_type) + "/" + String(i);
            relay_config["optimistic"] = false;
            relay_config["payload_on"] = "1";
            relay_config["payload_off"] = "0";
            relay_config["state_on"] = "1";
            relay_config["state_off"] = "0";
            relay_config["schema"] = "basic";
            
            // Add device info
            send_device_info(relay_config);
            
            String relay_config_str;
            serializeJson(relay_config, relay_config_str);
            client.setBufferSize(relay_config_str.length() + 64);
            // Config topic format: device_topic_{index}/config (matches main branch)
            client.publish(String(device_topic + "_" + String(i) + "/config").c_str(), relay_config_str.c_str());
            client.setBufferSize(MQTT_MAX_PACKET_SIZE);
            debug_printf("Sent relay config for pin %d", i);
        }
    }
#elif !defined(BRI_CONTROL) && !defined(CCT_CONTROL)
    if (conf_id == 0)
        for (size_t i = 1; i < PinsCount; i++)
            send_config(i);
#endif
}

// Универсальная отправка информации о устройстве
void send_device_info(JsonDocument &config) {
    config["device"]["name"] = dev_name;
    config["device"]["model"] = String(vendor_info.model) + " " + String(dev_name);
    config["device"]["hw_version"] = vendor_info.hw_version;
    config["device"]["sw_version"] = vendor_info.sw_version;
    config["device"]["manufacturer"] = vendor_info.manufacturer;

    JsonArray connections = config["device"].createNestedArray("connections");
    JsonArray mac = connections.createNestedArray();
    JsonArray ip = connections.createNestedArray();
    mac.add("mac");
    mac.add(WiFi.macAddress());
    ip.add("ip");
    ip.add(WiFi.localIP().toString());

    config["device"]["identifiers"] = get_sta_mac();
}

// Отправка конфигурации датчиков
void send_sensor_config() {
    return send_sensor_config(0);
}

void send_sensor_config(unsigned conf_id) {
    if (!client.connected())
        return;

    JsonDocument config;
    config["unique_id"] =
            endpoint_type + String("_") + get_sta_mac() + "_sensor_" + String(sensors_conf[conf_id].default_name);
    config["device"]["name"] = dev_name;
    config["device"]["model"] = String(vendor_info.model) + " " + String(dev_name);
    config["device"]["hw_version"] = vendor_info.hw_version;
    config["device"]["sw_version"] = vendor_info.sw_version;
    //config["support_url"] = vendor_info.support_url;
    config["device"]["manufacturer"] = vendor_info.manufacturer;
    JsonArray connections = config["device"].createNestedArray("connections");

    JsonArray mac = connections.createNestedArray();
    JsonArray ip = connections.createNestedArray();
    mac.add("mac");
    mac.add(WiFi.macAddress());
    ip.add("ip");
    ip.add(WiFi.localIP().toString());

    config["device"]["identifiers"] = get_sta_mac();
    config["state_topic"] = device_sensor_topic + "/state";
    config["name"] = sensors_conf[conf_id].default_name;
    config["device_class"] = sensors_conf[conf_id].device_class;
    config["state_class"] = sensors_conf[conf_id].state_class;
    config["unit_of_measurement"] = sensors_conf[conf_id].unit_of_measurement;

    if (sensors_conf[conf_id].diag)
        config["entity_category"] = "diagnostic";
    if (sensors_conf[conf_id].hide)
        config["enabled_by_default"] = false;

    config["value_template"] = String("{{ value_json.") + String(sensors_conf[conf_id].default_name) + String(" }}");

    String config_str;
    serializeJson(config, config_str);
    String sensor_config_topic = String(
            device_sensor_topic + "_" + String(sensors_conf[conf_id].default_name) + "/config");
    debug_printf("Size of sensor_config[%u]: %u, send config to mqtt", conf_id, config_str.length());
    client.setBufferSize(config_str.length() + 64 + sensor_config_topic.length());
    client.publish(sensor_config_topic.c_str(), config_str.c_str());
    client.setBufferSize(MQTT_MAX_PACKET_SIZE);

    if (conf_id == 0 && sensors_count > 1)
        for (unsigned i = 1; i < sensors_count; i++)
            send_sensor_config(i);
}

// Установка флага отправки данных
void status_send_f() {
    send_status_flag = true;
}

// Установка флага отправки данных с датчиков
void sensors_send_f() {
    send_sensor_flag = true;
}
