# Unit Tests for ESP8266 Light Control

This directory contains comprehensive unit tests for all functions and methods in the light control ESP8266 project.

## Test Structure

The tests are organized into modules matching the source code structure:

- **test_main_logic/** - Tests for I/O control functions (main_logic.ino)
  - Pin state management
  - PWM control
  - Power management
  - CCT (color temperature) control
  - Brightness control

- **test_utils/** - Tests for utility functions (utils.ino)
  - MQTT status publishing
  - I/O state publishing  
  - Device configuration
  - Sensor configuration
  - Helper functions

- **test_mqtt/** - Tests for MQTT message handlers (mqtt_utils.ino)
  - Status message handling
  - I/O command handling
  - CCT control commands
  - Brightness control commands
  - Power commands

- **test_sensors/** - Tests for sensor functions (sens_logic.ino)
  - Sensor initialization
  - Sensor index lookup
  - Sensor value management
  - Sensor metadata

## Running Tests

### Prerequisites

- CMake 3.10 or later
- g++ compiler (C++11 or later)
- Internet connection (for downloading Unity test framework on first build)

### Build and Run

```bash
# Create build directory
mkdir build && cd build

# Configure with CMake
cmake ..

# Build all tests
make

# Run all tests
ctest --output-on-failure

# Or run individual tests
./test_main_logic
./test_utils
./test_mqtt
./test_sensors
```

## Test Framework

Tests use the [Unity](https://github.com/ThrowTheSwitch/Unity) test framework, which is automatically downloaded during the first build.

## Mock Objects

The `test/mocks/` directory contains mock implementations of Arduino and ESP8266 libraries:

- **Arduino.h** - Core Arduino functions and types
- **ESP8266WiFi.h** - WiFi functionality  
- **PubSubClient.h** - MQTT client
- **ArduinoJson.h** - JSON serialization
- **FileData.h** - File system operations
- **Ticker.h** - Timer functionality

These mocks allow the tests to run on the development machine without requiring actual ESP8266 hardware.

## Test Coverage

The test suite covers:

- ✅ 19 tests for main logic functions (set_out, set_out_pwm, get_power, set_power, etc.)
- ✅ 14 tests for utility functions (send_status, send_io, send_power, send_cct, etc.)
- ✅ 18 tests for MQTT handlers (handle_cct_control, handle_brightness_control, etc.)
- ✅ 15 tests for sensor functions (init_all_sensors, get_sensor_index, etc.)

**Total: 66 unit tests**

All tests validate both normal operation and edge cases (invalid inputs, boundary conditions, etc.).

## Adding New Tests

To add new tests:

1. Create a new test file in the appropriate test directory (e.g., `test/test_mymodule/test_mymodule.cpp`)
2. Include Unity and test helpers:
   ```cpp
   #include <unity.h>
   #include "test_helpers.h"
   ```
3. Write test functions using Unity assertions:
   ```cpp
   void test_my_function(void) {
       // Setup
       int result = my_function(42);
       
       // Assert
       TEST_ASSERT_EQUAL(expected, result);
   }
   ```
4. Register tests in `main()`:
   ```cpp
   int main(int argc, char **argv) {
       UNITY_BEGIN();
       RUN_TEST(test_my_function);
       return UNITY_END();
   }
   ```
5. Update CMakeLists.txt to include your new test executable

## PlatformIO Alternative

Tests can also be run using PlatformIO (if configured):

```bash
pio test -e native
```

The `platformio.ini` file is already configured for native testing.

## Continuous Integration

Tests are designed to be integrated into CI/CD pipelines. Example GitLab CI configuration:

```yaml
test:
  stage: test
  script:
    - mkdir build && cd build
    - cmake ..
    - make
    - ctest --output-on-failure
  tags:
    - shell
```

## Notes

- Tests are designed to run quickly and independently
- Each test has its own setUp() and tearDown() for isolation
- Mocks are lightweight and focused on test needs
- No hardware or external dependencies required
