#ifndef ARDUINO_MOCK_H
#define ARDUINO_MOCK_H

#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <cstddef>

// Arduino basic types
typedef bool boolean;
typedef uint8_t byte;

// Pin modes
#define INPUT 0x0
#define OUTPUT 0x1
#define INPUT_PULLUP 0x2

// Pin states
#ifndef HIGH
#define HIGH 0x1
#endif

#ifndef LOW
#define LOW 0x0
#endif

// PROGMEM support (for ArduinoJson compatibility)
#define PROGMEM
#define PGM_P const char *
#define PSTR(s) (s)

// Forward declaration
class __FlashStringHelper;

// PROGMEM access functions
inline uint8_t pgm_read_byte(const void* addr) {
    return *(const uint8_t*)addr;
}

inline uint16_t pgm_read_word(const void* addr) {
    return *(const uint16_t*)addr;
}

inline uint32_t pgm_read_dword(const void* addr) {
    return *(const uint32_t*)addr;
}

inline void* pgm_read_ptr(const void* addr) {
    return *(void* const *)addr;
}

inline float pgm_read_float(const void* addr) {
    return *(const float*)addr;
}

// FlashStringHelper
class __FlashStringHelper;
#define F(string_literal) (reinterpret_cast<const __FlashStringHelper *>(string_literal))

// Pin definitions for ESP8266
#define D0 16
#define D1 5
#define D2 4
#define D3 0
#define D4 2
#define D5 14
#define D6 12
#define D7 13
#define D8 15

// String class mock
class String {
public:
    String() : str(nullptr), len(0) {}
    String(const char* s) {
        if (s) {
            len = strlen(s);
            str = (char*)malloc(len + 1);
            strcpy(str, s);
        } else {
            str = nullptr;
            len = 0;
        }
    }
    String(const String& s) {
        len = s.len;
        if (s.str) {
            str = (char*)malloc(len + 1);
            strcpy(str, s.str);
        } else {
            str = nullptr;
        }
    }
    String(int val) {
        char buffer[32];
        snprintf(buffer, sizeof(buffer), "%d", val);
        len = strlen(buffer);
        str = (char*)malloc(len + 1);
        strcpy(str, buffer);
    }
    String(unsigned int val) {
        char buffer[32];
        snprintf(buffer, sizeof(buffer), "%u", val);
        len = strlen(buffer);
        str = (char*)malloc(len + 1);
        strcpy(str, buffer);
    }
    String(unsigned long val) {
        char buffer[32];
        snprintf(buffer, sizeof(buffer), "%lu", val);
        len = strlen(buffer);
        str = (char*)malloc(len + 1);
        strcpy(str, buffer);
    }
    String(float val, int decimals = 2) {
        char buffer[32];
        snprintf(buffer, sizeof(buffer), "%.*f", decimals, val);
        len = strlen(buffer);
        str = (char*)malloc(len + 1);
        strcpy(str, buffer);
    }
    ~String() {
        if (str) free(str);
    }
    
    String& operator=(const String& s) {
        if (this != &s) {
            if (str) free(str);
            len = s.len;
            if (s.str) {
                str = (char*)malloc(len + 1);
                strcpy(str, s.str);
            } else {
                str = nullptr;
            }
        }
        return *this;
    }
    
    String& operator=(const char* s) {
        if (str) free(str);
        if (s) {
            len = strlen(s);
            str = (char*)malloc(len + 1);
            strcpy(str, s);
        } else {
            str = nullptr;
            len = 0;
        }
        return *this;
    }
    
    String operator+(const String& s) const {
        String result;
        result.len = len + s.len;
        result.str = (char*)malloc(result.len + 1);
        if (str) strcpy(result.str, str);
        else result.str[0] = '\0';
        if (s.str) strcat(result.str, s.str);
        return result;
    }
    
    String operator+(const char* s) const {
        String result;
        result.len = len + (s ? strlen(s) : 0);
        result.str = (char*)malloc(result.len + 1);
        if (str) strcpy(result.str, str);
        else result.str[0] = '\0';
        if (s) strcat(result.str, s);
        return result;
    }
    
    friend String operator+(const char* lhs, const String& rhs) {
        String result;
        result.len = (lhs ? strlen(lhs) : 0) + rhs.len;
        result.str = (char*)malloc(result.len + 1);
        if (lhs) strcpy(result.str, lhs);
        else result.str[0] = '\0';
        if (rhs.str) strcat(result.str, rhs.str);
        return result;
    }
    
    bool operator==(const String& s) const {
        if (!str && !s.str) return true;
        if (!str || !s.str) return false;
        return strcmp(str, s.str) == 0;
    }
    
    bool operator==(const char* s) const {
        if (!str && !s) return true;
        if (!str || !s) return false;
        return strcmp(str, s) == 0;
    }
    
    bool operator<(const String& s) const {
        if (!str && !s.str) return false;
        if (!str) return true;
        if (!s.str) return false;
        return strcmp(str, s.str) < 0;
    }
    
    const char* c_str() const { return str ? str : ""; }
    size_t length() const { return len; }
    
    int toInt() const {
        return str ? atoi(str) : 0;
    }
    
    String substring(size_t start, size_t end) const {
        if (!str || start >= len) return String();
        if (end > len) end = len;
        size_t sublen = end - start;
        String result;
        result.len = sublen;
        result.str = (char*)malloc(sublen + 1);
        strncpy(result.str, str + start, sublen);
        result.str[sublen] = '\0';
        return result;
    }
    
    String substring(size_t start) const {
        return substring(start, len);
    }
    
    int indexOf(char ch) const {
        if (!str) return -1;
        const char* p = strchr(str, ch);
        return p ? (p - str) : -1;
    }
    
    int indexOf(const char* substr) const {
        if (!str || !substr) return -1;
        const char* p = strstr(str, substr);
        return p ? (p - str) : -1;
    }
    
    int indexOf(const String& substr) const {
        return indexOf(substr.c_str());
    }
    
    bool startsWith(const String& prefix) const {
        if (!str || !prefix.str) return false;
        return strncmp(str, prefix.str, prefix.len) == 0;
    }
    
    bool equals(const String& s) const {
        return *this == s;
    }
    
    char charAt(unsigned int index) const {
        if (!str || index >= len) return 0;
        return str[index];
    }
    
    char operator[](unsigned int index) const {
        return charAt(index);
    }
    
    // concat method for ArduinoJson compatibility
    unsigned char concat(const char* cstr) {
        if (!cstr) return 0;
        size_t clen = strlen(cstr);
        char* newstr = (char*)malloc(len + clen + 1);
        if (str) strcpy(newstr, str);
        else newstr[0] = '\0';
        strcat(newstr, cstr);
        if (str) free(str);
        str = newstr;
        len += clen;
        return 1;
    }
    
    // operator+= for char (efficient - direct append)
    String& operator+=(char c) {
        char* newstr = (char*)malloc(len + 2);
        if (str) {
            strcpy(newstr, str);
            free(str);
        } else {
            newstr[0] = '\0';
        }
        newstr[len] = c;
        newstr[len + 1] = '\0';
        str = newstr;
        len++;
        return *this;
    }
    
    // operator+= for String (efficient - direct append)
    String& operator+=(const String& s) {
        if (!s.str || s.len == 0) return *this;
        char* newstr = (char*)malloc(len + s.len + 1);
        if (str) {
            strcpy(newstr, str);
            free(str);
        } else {
            newstr[0] = '\0';
        }
        strcat(newstr, s.str);
        str = newstr;
        len += s.len;
        return *this;
    }
    
    // operator+= for const char* (efficient - direct append)
    String& operator+=(const char* s) {
        if (!s) return *this;
        size_t slen = strlen(s);
        if (slen == 0) return *this;
        char* newstr = (char*)malloc(len + slen + 1);
        if (str) {
            strcpy(newstr, str);
            free(str);
        } else {
            newstr[0] = '\0';
        }
        strcat(newstr, s);
        str = newstr;
        len += slen;
        return *this;
    }
    
private:
    char* str;
    size_t len;
};

// IPAddress class mock
class IPAddress {
public:
    IPAddress() : _address{0,0,0,0} {}
    IPAddress(uint8_t a, uint8_t b, uint8_t c, uint8_t d) : _address{a,b,c,d} {}
    
    String toString() const {
        char buffer[16];
        snprintf(buffer, sizeof(buffer), "%d.%d.%d.%d", 
                 _address[0], _address[1], _address[2], _address[3]);
        return String(buffer);
    }
    
private:
    uint8_t _address[4];
};

// Mock global variables
extern unsigned long g_millis;
extern unsigned long g_micros;

// Mock functions
inline void pinMode(uint8_t pin, uint8_t mode) {}
inline void digitalWrite(uint8_t pin, uint8_t val) {}
inline int digitalRead(uint8_t pin) { return LOW; }
inline void analogWrite(uint8_t pin, int val) {}
inline int analogRead(uint8_t pin) { return 0; }
inline unsigned long millis() { return g_millis; }
inline unsigned long micros() { return g_micros; }
inline void delay(unsigned long ms) { g_millis += ms; }
inline void delayMicroseconds(unsigned int us) { g_micros += us; }
inline void yield() {}

// Print and Printable classes (for ArduinoJson compatibility)
class Print {
public:
    virtual ~Print() {}
    virtual size_t write(uint8_t) = 0;
    virtual size_t write(const uint8_t *buffer, size_t size) {
        size_t n = 0;
        while (size--) {
            n += write(*buffer++);
        }
        return n;
    }
    size_t write(const char *str) {
        if (str == NULL) return 0;
        return write((const uint8_t *)str, strlen(str));
    }
    size_t print(const char* s) { return write(s); }
    size_t print(const String& s) { return write(s.c_str()); }
    size_t println(const char* s = "") { return write(s) + write("\n"); }
};

class Printable {
public:
    virtual ~Printable() {}
    virtual size_t printTo(Print& p) const = 0;
};

class Stream : public Print {
public:
    virtual int available() = 0;
    virtual int read() = 0;
    virtual int peek() = 0;
    size_t write(uint8_t c) override { return 1; }
    
    // readBytes for ArduinoJson compatibility
    size_t readBytes(char *buffer, size_t length) {
        size_t count = 0;
        while (count < length) {
            int c = read();
            if (c < 0) break;
            *buffer++ = (char)c;
            count++;
        }
        return count;
    }
};

// Add concat() method to String class for ArduinoJson
inline unsigned char String_concat(String* self, const char* cstr) {
    if (!cstr) return 0;
    *self = *self + cstr;
    return 1;
}

// Serial mock
class SerialMock : public Print {
public:
    void begin(unsigned long baud) {}
    size_t write(uint8_t c) override { return 1; }
    size_t print(const char* s) { return s ? strlen(s) : 0; }
    size_t print(const String& s) { return s.length(); }
    size_t print(char c) { return 1; }
    size_t print(int n) { return 1; }
    size_t println(const char* s = "") { return print(s) + 1; }
    size_t println(const String& s) { return s.length() + 1; }
    size_t println(int n) { return 2; }
    int available() { return 0; }
    int read() { return -1; }
    int peek() { return -1; }
};

extern SerialMock Serial;

// Random
inline long random(long min, long max) {
    return min + (rand() % (max - min));
}

inline long random(long max) {
    return rand() % max;
}

#endif // ARDUINO_MOCK_H
