#ifndef ESP8266_WIFI_MOCK_H
#define ESP8266_WIFI_MOCK_H

#include "Arduino.h"

// WiFi status codes
#define WL_IDLE_STATUS 0
#define WL_NO_SSID_AVAIL 1
#define WL_SCAN_COMPLETED 2
#define WL_CONNECTED 3
#define WL_CONNECT_FAILED 4
#define WL_CONNECTION_LOST 5
#define WL_DISCONNECTED 6

// WiFi modes
#define WIFI_STA 1
#define WIFI_AP 2
#define WIFI_AP_STA 3

// Station interface
#define STATION_IF 0
#define SOFTAP_IF 1

class WiFiClass {
public:
    int status() { return _status; }
    void mode(int m) { _mode = m; }
    void hostname(const char* name) { _hostname = name; }
    void begin(const char* ssid, const char* pass) {
        _ssid = ssid;
        _password = pass;
        _status = WL_CONNECTED;
    }
    IPAddress localIP() { return _localIP; }
    String SSID() { return _ssid; }
    int RSSI() { return _rssi; }
    String macAddress() { return _macAddress; }
    void disconnect() { _status = WL_DISCONNECTED; }
    void reconnect() { _status = WL_CONNECTED; }
    
    // For testing
    void setStatus(int status) { _status = status; }
    void setLocalIP(uint8_t a, uint8_t b, uint8_t c, uint8_t d) {
        _localIP = IPAddress(a, b, c, d);
    }
    void setLocalIP(const char* ip) {
        // Parse IP string like "192.168.4.139"
        int a, b, c, d;
        sscanf(ip, "%d.%d.%d.%d", &a, &b, &c, &d);
        _localIP = IPAddress(a, b, c, d);
    }
    void setRSSI(int rssi) { _rssi = rssi; }
    void setMacAddress(const char* mac) { _macAddress = mac; }
    
private:
    int _status = WL_DISCONNECTED;
    int _mode = WIFI_STA;
    String _hostname;
    String _ssid;
    String _password;
    IPAddress _localIP;
    int _rssi = -70;
    String _macAddress = "AA:BB:CC:DD:EE:FF";
};

extern WiFiClass WiFi;

// Mock WiFi functions
inline void wifi_get_macaddr(int mode, uint8_t* mac) {
    // Mock MAC address: AA:BB:CC:DD:EE:FF
    mac[0] = 0xAA;
    mac[1] = 0xBB;
    mac[2] = 0xCC;
    mac[3] = 0xDD;
    mac[4] = 0xEE;
    mac[5] = 0xFF;
}

// WiFiClient mock
class WiFiClient {
public:
    bool connected() { return _connected; }
    void setConnected(bool c) { _connected = c; }
    
private:
    bool _connected = false;
};

// WiFiUDP mock
class WiFiUDP {
public:
    void beginPacket(IPAddress ip, uint16_t port) {}
    void write(const char* buf) {}
    void print(const char* buf) {}
    void endPacket() {}
};

#endif // ESP8266_WIFI_MOCK_H
