#ifndef PUBSUB_CLIENT_MOCK_H
#define PUBSUB_CLIENT_MOCK_H

#include "Arduino.h"
#include "ESP8266WiFi.h"
#include <vector>
#include <functional>

#define MQTT_MAX_PACKET_SIZE 512

typedef std::function<void(char*, uint8_t*, unsigned int)> MQTT_CALLBACK_SIGNATURE;

struct PublishedMessage {
    String topic;
    String payload;
    bool retained;
};

class PubSubClient {
public:
    PubSubClient(WiFiClient& client) : _client(client) {}
    
    void setServer(const char* server, uint16_t port) {
        _server = server;
        _port = port;
    }
    
    void setCallback(MQTT_CALLBACK_SIGNATURE callback) {
        _callback = callback;
    }
    
    bool connect(const char* id, const char* user = nullptr, const char* pass = nullptr) {
        _clientId = id;
        _connected = true;
        return true;
    }
    
    void disconnect() {
        _connected = false;
    }
    
    bool connected() {
        return _connected;
    }
    
    bool publish(const char* topic, const char* payload, bool retained = false) {
        if (!_connected) return false;
        _lastPublishTopic = topic;
        _lastPublishPayload = payload;
        _publishedMessages.push_back({String(topic), String(payload), retained});
        return true;
    }
    
    bool subscribe(const char* topic) {
        if (!_connected) return false;
        _subscriptions.push_back(topic);
        return true;
    }
    
    void loop() {
        // Process any pending messages
    }
    
    int state() {
        return _connected ? 0 : -1;
    }
    
    void setBufferSize(uint16_t size) {
        _bufferSize = size;
    }
    
    // Test helpers
    void simulateMessage(const char* topic, const char* payload) {
        if (_callback) {
            _callback((char*)topic, (uint8_t*)payload, strlen(payload));
        }
    }
    
    String getLastPublishTopic() const { return _lastPublishTopic; }
    String getLastPublishPayload() const { return _lastPublishPayload; }
    const std::vector<String>& getSubscriptions() const { return _subscriptions; }
    void setConnected(bool connected) { _connected = connected; }
    
    // Integration test helpers
    const std::vector<PublishedMessage>& getPublishedMessages() const { return _publishedMessages; }
    void clearPublished() { _publishedMessages.clear(); }
    uint16_t getBufferSize() const { return _bufferSize; }
    
private:
    WiFiClient& _client;
    String _server;
    uint16_t _port = 1883;
    String _clientId;
    bool _connected = false;
    MQTT_CALLBACK_SIGNATURE _callback;
    std::vector<String> _subscriptions;
    String _lastPublishTopic;
    String _lastPublishPayload;
    uint16_t _bufferSize = MQTT_MAX_PACKET_SIZE;
    std::vector<PublishedMessage> _publishedMessages;
};

#endif // PUBSUB_CLIENT_MOCK_H
