/**
 * Common Test Infrastructure for Home Assistant MQTT Auto-Discovery Tests
 * 
 * This header provides shared test fixtures, helper functions, and global
 * variables used across all HA integration test files.
 */

#ifndef TEST_HA_COMMON_H
#define TEST_HA_COMMON_H

#include <gtest/gtest.h>
#include <cstdlib>
#include <ctime>
#include <algorithm>
#include "../lamp_sources.h"
#include "../test_helpers.h"

// ============================================================================
// Global Test Variables
// ============================================================================

extern size_t PinsCount;
extern pin_state_str pin_state[];
extern main_state_str main_state;
extern FileData<FSMock> pinStatesFile;
extern FileData<FSMock> mainStateFile;
extern WiFiClient espClient;
extern PubSubClient client;
extern WiFiUDP udp;
extern String device_topic;
extern String device_sensor_topic;
extern String cmd_topic;
extern String state_topic;
extern const char* dev_name;
extern const char* endpoint_type;
extern const char* topic_prefix;
extern const char* device_type;
extern vendor_info_str vendor_info;
extern sensor_state_str sensors_conf[];
extern size_t sensors_count;
extern volatile bool send_status_flag;
extern volatile bool send_sensor_flag;
extern unsigned long previousMillis;
extern String clientId;
extern const char* mqtt_user;
extern const char* mqtt_password;

// ============================================================================
// Helper Functions
// ============================================================================

// Helper function to generate random MAC address
inline String generateRandomMAC() {
    char mac[18];
    unsigned int seed = static_cast<unsigned int>(time(nullptr)) + static_cast<unsigned int>(clock());
    srand(seed);
    
    int byte1 = rand() % 256;
    int byte2 = rand() % 256;
    int byte3 = rand() % 256;
    int byte4 = rand() % 256;
    int byte5 = rand() % 256;
    int byte6 = rand() % 256;
    
    sprintf(mac, "%02X:%02X:%02X:%02X:%02X:%02X", byte1, byte2, byte3, byte4, byte5, byte6);
    return String(mac);
}

// ============================================================================
// Base Test Fixture
// ============================================================================

class HAIntegrationTestBase : public ::testing::Test {
protected:
    String test_mac;
    
    void SetUp() override {
        reset_test_state();
        
        test_mac = generateRandomMAC();
        WiFi.setMacAddress(test_mac.c_str());
        WiFi.setLocalIP("192.168.4.139");
        
        client.setConnected(true);
        
        String mac = get_sta_mac();
        device_topic = String(topic_prefix) + "/" + String(device_type) + "/" + String(endpoint_type) + "_" + mac;
        cmd_topic = device_topic + "/cmd";
        state_topic = device_topic + "/state";
        device_sensor_topic = String(topic_prefix) + "/sensor/" + String(endpoint_type) + "_" + mac;
    }

    void TearDown() override {
        client.clearPublished();
    }
    
    String getFormattedMAC() {
        return get_sta_mac();
    }
    
    String getLightConfigTopic() {
        return String(topic_prefix) + "/light/" + endpoint_type + "_" + getFormattedMAC() + "/config";
    }
    
    String getSensorConfigTopic(const char* sensor_name) {
        return String(topic_prefix) + "/sensor/" + endpoint_type + "_" + getFormattedMAC() + "_" + sensor_name + "/config";
    }
    
    bool wasTopicPublished(const String& topic) {
        auto published = client.getPublishedMessages();
        for (const auto& msg : published) {
            if (msg.topic == topic) {
                return true;
            }
        }
        return false;
    }
    
    String getPublishedPayload(const String& topic) {
        auto published = client.getPublishedMessages();
        for (const auto& msg : published) {
            if (msg.topic == topic) {
                return msg.payload;
            }
        }
        return "";
    }
    
    bool payloadContains(const String& topic, const String& substring) {
        String payload = getPublishedPayload(topic);
        return payload.indexOf(substring) >= 0;
    }
    
    // Helper to find char starting from a specific position
    int indexOfFrom(const String& str, char ch, int fromPos) {
        if (fromPos >= (int)str.length()) return -1;
        for (int i = fromPos; i < (int)str.length(); i++) {
            if (str[i] == ch) return i;
        }
        return -1;
    }
    
    // Extract JSON value by key - simple helper for testing
    String extractJsonValue(const String& payload, const String& key) {
        String searchKey = "\"" + key + "\"";
        int keyPos = payload.indexOf(searchKey);
        if (keyPos < 0) return "";
        
        int colonPos = indexOfFrom(payload, ':', keyPos);
        if (colonPos < 0) return "";
        
        int valueStart = colonPos + 1;
        while (valueStart < (int)payload.length() && (payload[valueStart] == ' ' || payload[valueStart] == '\t')) {
            valueStart++;
        }
        
        if (valueStart >= (int)payload.length()) return "";
        
        char startChar = payload[valueStart];
        if (startChar == '"') {
            int endQuote = indexOfFrom(payload, '"', valueStart + 1);
            if (endQuote < 0) return "";
            return payload.substring(valueStart + 1, endQuote);
        } else if (startChar == '[') {
            int depth = 1;
            int endBracket = valueStart + 1;
            while (endBracket < (int)payload.length() && depth > 0) {
                if (payload[endBracket] == '[') depth++;
                if (payload[endBracket] == ']') depth--;
                endBracket++;
            }
            return payload.substring(valueStart, endBracket);
        } else if (startChar == '{') {
            int depth = 1;
            int endBrace = valueStart + 1;
            while (endBrace < (int)payload.length() && depth > 0) {
                if (payload[endBrace] == '{') depth++;
                if (payload[endBrace] == '}') depth--;
                endBrace++;
            }
            return payload.substring(valueStart, endBrace);
        } else {
            int endPos = valueStart;
            while (endPos < (int)payload.length() && 
                   payload[endPos] != ',' && 
                   payload[endPos] != '}' &&
                   payload[endPos] != ']') {
                endPos++;
            }
            String value = payload.substring(valueStart, endPos);
            while (value.length() > 0 && (value[0] == ' ' || value[0] == '\t')) {
                value = value.substring(1);
            }
            while (value.length() > 0 && (value[value.length()-1] == ' ' || value[value.length()-1] == '\t')) {
                value = value.substring(0, value.length()-1);
            }
            return value;
        }
    }
};

#endif // TEST_HA_COMMON_H
