/**
 * Hardware Configuration Tests
 * 
 * Parameterized tests for all hardware configurations from the configs/ directory.
 * Tests device type, endpoint type, brightness control, and CCT features.
 */

#include "test_ha_common.h"
#include <algorithm>

// ============================================================================
// Configuration Test Parameters
// ============================================================================

struct ConfigTestParams {
    const char* config_name;
    const char* expected_device_type;
    const char* expected_endpoint_type;
    bool has_bri_control;
    bool has_pwm_control;
    bool has_cct_control;
    int cct_start;
    int cct_end;
    bool has_pulse_control;
    
    // Human-readable description for test output
    friend std::ostream& operator<<(std::ostream& os, const ConfigTestParams& params) {
        os << params.config_name 
           << " [type=" << params.expected_device_type 
           << ", endpoint=" << params.expected_endpoint_type;
        if (params.has_bri_control) os << ", BRI";
        if (params.has_pwm_control) os << ", PWM";
        if (params.has_cct_control) os << ", CCT(" << params.cct_start << "-" << params.cct_end << "K)";
        if (params.has_pulse_control) os << ", PULSE";
        os << "]";
        return os;
    }
};

class HAConfigTest : public ::testing::TestWithParam<ConfigTestParams> {
protected:
    String test_mac;
    const char* saved_device_type;
    const char* saved_endpoint_type;
    
    void SetUp() override {
        test_mac = generateRandomMAC();
        WiFi.setMacAddress(test_mac.c_str());
        WiFi.setLocalIP("192.168.4.139");
        client.setServer("192.168.4.197", 1883);
        client.setConnected(true);
        client.clearPublished();
        
        // Save original values
        saved_device_type = device_type;
        saved_endpoint_type = endpoint_type;
        
        // Set runtime configuration from test parameters
        device_type = GetParam().expected_device_type;
        endpoint_type = GetParam().expected_endpoint_type;
        
        // Reinitialize topics with new configuration
        String mac = get_sta_mac();
        device_topic = String(topic_prefix) + "/" + String(device_type) + "/" + String(endpoint_type) + "_" + mac;
        cmd_topic = device_topic + "/cmd";
        state_topic = device_topic + "/state";
        device_sensor_topic = String(topic_prefix) + "/sensor/" + String(endpoint_type) + "_" + mac;
    }
    
    void TearDown() override {
        client.clearPublished();
        // Restore original values
        device_type = saved_device_type;
        endpoint_type = saved_endpoint_type;
    }
    
    String getFormattedMAC() {
        // Remove colons from MAC address
        String result;
        for (unsigned int i = 0; i < test_mac.length(); i++) {
            if (test_mac[i] != ':') {
                result += test_mac[i];
            }
        }
        return result;
    }
    
    String getLightConfigTopic() {
        return String(topic_prefix) + "/" + GetParam().expected_device_type + "/" + GetParam().expected_endpoint_type + "_" + getFormattedMAC() + "/config";
    }
    
    // Helper: Build base pattern for config topic
    // Pattern: {topic_prefix}/{device_type}/{endpoint_type}_{MAC}
    String buildBasePattern() {
        return String(topic_prefix) + "/" + GetParam().expected_device_type + "/" + GetParam().expected_endpoint_type + "_" + getFormattedMAC();
    }
    
    // Helper: Build expected device topic
    // Pattern: {topic_prefix}/{device_type}/{endpoint_type}_{MAC}
    String buildExpectedDeviceTopic() {
        return buildBasePattern();
    }
    
    // Helper: Check if string ends with suffix
    bool stringEndsWith(const String& str, const String& suffix) {
        return (str.length() >= suffix.length()) && 
               (str.substring(str.length() - suffix.length()) == suffix);
    }
    
    // Helper: Check if character is valid for identifiers
    bool isValidIdentifierChar(char c) {
        return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || 
               (c >= '0' && c <= '9') || c == '_';
    }
    
    bool wasTopicPublished(const String& topic) {
        auto published = client.getPublishedMessages();
        for (const auto& msg : published) {
            if (msg.topic == topic) {
                return true;
            }
        }
        return false;
    }
    
    String getPublishedPayload(const String& topic) {
        auto published = client.getPublishedMessages();
        for (const auto& msg : published) {
            if (msg.topic == topic) {
                return msg.payload;
            }
        }
        return "";
    }
    
    bool payloadContains(const String& topic, const String& substring) {
        String payload = getPublishedPayload(topic);
        return payload.indexOf(substring) >= 0;
    }
    
    // Helper to find char starting from a specific position
    int indexOfFrom(const String& str, char ch, int fromPos) {
        if (fromPos >= (int)str.length()) return -1;
        for (int i = fromPos; i < (int)str.length(); i++) {
            if (str[i] == ch) return i;
        }
        return -1;
    }
    
    // Extract JSON value by key - simple helper for testing
    String extractJsonValue(const String& payload, const String& key) {
        String searchKey = "\"" + key + "\"";
        int keyPos = payload.indexOf(searchKey);
        if (keyPos < 0) return "";
        
        int colonPos = indexOfFrom(payload, ':', keyPos);
        if (colonPos < 0) return "";
        
        int valueStart = colonPos + 1;
        while (valueStart < (int)payload.length() && (payload[valueStart] == ' ' || payload[valueStart] == '\t')) {
            valueStart++;
        }
        
        if (valueStart >= (int)payload.length()) return "";
        
        char startChar = payload[valueStart];
        if (startChar == '"') {
            int endQuote = indexOfFrom(payload, '"', valueStart + 1);
            if (endQuote < 0) return "";
            return payload.substring(valueStart + 1, endQuote);
        } else if (startChar == '[') {
            int depth = 1;
            int endBracket = valueStart + 1;
            while (endBracket < (int)payload.length() && depth > 0) {
                if (payload[endBracket] == '[') depth++;
                if (payload[endBracket] == ']') depth--;
                endBracket++;
            }
            return payload.substring(valueStart, endBracket);
        } else if (startChar == '{') {
            int depth = 1;
            int endBrace = valueStart + 1;
            while (endBrace < (int)payload.length() && depth > 0) {
                if (payload[endBrace] == '{') depth++;
                if (payload[endBrace] == '}') depth--;
                endBrace++;
            }
            return payload.substring(valueStart, endBrace);
        } else {
            int endPos = valueStart;
            while (endPos < (int)payload.length() && 
                   payload[endPos] != ',' && 
                   payload[endPos] != '}' &&
                   payload[endPos] != ']') {
                endPos++;
            }
            String value = payload.substring(valueStart, endPos);
            while (value.length() > 0 && (value[0] == ' ' || value[0] == '\t')) {
                value = value.substring(1);
            }
            while (value.length() > 0 && (value[value.length()-1] == ' ' || value[value.length()-1] == '\t')) {
                value = value.substring(0, value.length()-1);
            }
            return value;
        }
    }
};

// All 7 hardware configurations from the configs/ directory
const ConfigTestParams all_configs[] = {
    // config_lamp.h - Basic relay lamp with brightness control
    {"config_lamp.h", "light", "lamp", true, false, false, 0, 0, false},
    
    // config_lamp_pulse.h - Pulse control lamp (current tests use this)
    {"config_lamp_pulse.h", "light", "lamp", false, false, false, 0, 0, true},
    
    // config_rgblamp.h - RGB WS2813 lamp
    {"config_rgblamp.h", "light", "lamp", false, false, false, 0, 0, false},
    
    // config_2_outlet.h - Power outlet/switch (NOT a light!)
    {"config_2_outlet.h", "switch", "plug", false, false, false, 0, 0, false},
    
    // config_led_strip.h - LED strip with PWM and brightness
    {"config_led_strip.h", "light", "strip", true, true, false, 0, 0, false},
    
    // config_led_lamp.h - LED lamp with CCT control (2700-5700K)
    {"config_led_lamp.h", "light", "lamp", true, true, true, 2700, 5700, false},
    
    // config_led_backlight_esp32.h - ESP32 backlight
    {"config_led_backlight_esp32.h", "light", "lamp", false, false, false, 0, 0, false},
};

INSTANTIATE_TEST_SUITE_P(
    AllConfigurations,
    HAConfigTest,
    ::testing::ValuesIn(all_configs),
    [](const ::testing::TestParamInfo<ConfigTestParams>& info) {
        std::string name = info.param.config_name;
        size_t dot_pos = name.find(".h");
        if (dot_pos != std::string::npos) {
            name = name.substr(0, dot_pos);
        }
        name.erase(std::remove(name.begin(), name.end(), '_'), name.end());
        name.erase(std::remove(name.begin(), name.end(), '.'), name.end());
        return name;
    }
);

// ============================================================================
// Parameterized Tests for Hardware Configurations
// ============================================================================

TEST_P(HAConfigTest, DeviceTypeCorrectInDiscovery) {
    auto params = GetParam();
    
    send_config();
    
    String config_topic = getLightConfigTopic();
    EXPECT_TRUE(wasTopicPublished(config_topic)) << "Config topic should be published for " << params.config_name;
    
    if (strcmp(params.expected_device_type, "switch") == 0) {
        String switch_topic = String(topic_prefix) + "/switch/lamp_" + getFormattedMAC() + "/config";
    }
}

TEST_P(HAConfigTest, EndpointTypeInTopicNaming) {
    auto params = GetParam();
    
    send_config();
    
    String mac = getFormattedMAC();
    auto messages = client.getPublishedMessages();
    
    bool found_endpoint = false;
    for (const auto& msg : messages) {
        if (msg.topic.indexOf(mac) >= 0) {
            found_endpoint = true;
            break;
        }
    }
    EXPECT_TRUE(found_endpoint) << "Should find MAC in published topics for " << params.config_name;
}

TEST_P(HAConfigTest, BrightnessControlFeature) {
    auto params = GetParam();
    
    send_config();
    
    String config_topic = getLightConfigTopic();
    if (!wasTopicPublished(config_topic)) {
        GTEST_SKIP() << "Config topic not published for " << params.config_name;
    }
    
    if (params.has_bri_control) {
        EXPECT_TRUE(payloadContains(config_topic, "brightness")) 
            << params.config_name << " should have brightness support";
    }
}

TEST_P(HAConfigTest, CCTRangeValidation) {
    auto params = GetParam();
    
    if (!params.has_cct_control) {
        GTEST_SKIP() << params.config_name << " does not have CCT control";
    }
    
    send_config();
    
    String config_topic = getLightConfigTopic();
    EXPECT_TRUE(wasTopicPublished(config_topic));
    
    if (params.cct_start == 2700 && params.cct_end == 5700) {
        EXPECT_TRUE(payloadContains(config_topic, "color_temp")) 
            << params.config_name << " should have color_temp in payload";
    }
}

TEST_P(HAConfigTest, RandomMACInAllConfigs) {
    auto params = GetParam();
    
    send_config();
    
    String mac = getFormattedMAC();
    String config_topic = getLightConfigTopic();
    
    if (wasTopicPublished(config_topic)) {
        EXPECT_TRUE(payloadContains(config_topic, mac.c_str())) 
            << params.config_name << " should contain MAC " << mac.c_str() << " in payload";
    }
}

// ============================================================================
// unique_id Format Consistency Tests
// ============================================================================

// This test verifies that all published entities follow the correct unique_id format.
// The format MUST be: {endpoint_type}_{MAC}_{index} (for indexed entities)
// or {endpoint_type}_{MAC} (for single entities)
//
// INCORRECT format that breaks backward compatibility:
// {endpoint_type}_relay_{index}_{MAC}
//
// This ensures that entities remain compatible between firmware versions
// and don't cause orphaned entities in Home Assistant.
TEST_P(HAConfigTest, UniqueIdFormatBackwardCompatibility) {
    auto params = GetParam();
    
    // Clear previous publishes
    client.clearPublished();
    
    // Publish configs
    send_config();
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    // Check all published config messages
    for (const auto& msg : published) {
        // Only check config topics
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String uniqueId = extractJsonValue(payload, "unique_id");
        String name = extractJsonValue(payload, "name");
        
        // Skip if no unique_id (shouldn't happen)
        if (uniqueId.length() == 0) continue;
        
        // TEST 1: unique_id should NOT contain "_relay_"
        // The word "relay" in unique_id breaks backward compatibility
        // with entities created by main branch firmware
        EXPECT_EQ(uniqueId.indexOf("_relay_"), -1)
            << params.config_name << ": unique_id '" << uniqueId.c_str() 
            << "' should NOT contain '_relay_'. "
            << "Expected format: {endpoint_type}_{MAC} or {endpoint_type}_{MAC}_{index}";
        
        // TEST 2: name should NOT contain "_relay_"
        EXPECT_EQ(name.indexOf("_relay_"), -1)
            << params.config_name << ": name '" << name.c_str() 
            << "' should NOT contain '_relay_'. "
            << "Expected format: {dev_name}_{MAC} or {dev_name}_{MAC}_{index}";
        
        // TEST 3: For indexed entities (ending with _N), MAC must come BEFORE the index
        // Check if unique_id ends with _N where N is a digit
        bool isIndexedEntity = false;
        int indexValue = -1;
        if (uniqueId.length() > 2) {
            char lastChar = uniqueId[uniqueId.length() - 1];
            char secondLastChar = uniqueId[uniqueId.length() - 2];
            if (lastChar >= '0' && lastChar <= '9' && secondLastChar == '_') {
                isIndexedEntity = true;
                indexValue = lastChar - '0';
            }
        }
        
        if (isIndexedEntity) {
            // Find MAC position in unique_id
            int macPos = uniqueId.indexOf(mac);
            
            // Find the position of the last underscore before the index digit
            int lastUnderscorePos = -1;
            for (int j = uniqueId.length() - 2; j >= 0; j--) {
                if (uniqueId[j] == '_') {
                    lastUnderscorePos = j;
                    break;
                }
            }
            
            // MAC should come BEFORE the index
            // Correct: lamp_083A8DCCC915_0 (macPos=5, indexPos=18)
            // Wrong:   lamp_relay_0_083A8DCCC915 (macPos=12, indexPos=10)
            if (macPos >= 0 && lastUnderscorePos >= 0) {
                EXPECT_LT(macPos, lastUnderscorePos)
                    << params.config_name << ": In unique_id '" << uniqueId.c_str() 
                    << "', MAC (" << mac.c_str() << ") should come BEFORE index " << indexValue
                    << ". MAC at position " << macPos << ", index at position " << lastUnderscorePos
                    << ". Expected format: {endpoint_type}_{MAC}_{index}";
            }
        }
        
        // TEST 4: unique_id MUST contain MAC address
        EXPECT_GE(uniqueId.indexOf(mac), 0)
            << params.config_name << ": unique_id '" << uniqueId.c_str() 
            << "' should contain MAC address '" << mac.c_str() << "'";
    }
}

// ============================================================================
// Topic Pattern Tests for ALL Configurations
// ============================================================================

// Pattern definitions for validation:
// - Config topic: {topic_prefix}/{device_type}/{endpoint_type}_{MAC}/config
//   or: {topic_prefix}/{device_type}/{endpoint_type}_{MAC}_{index}/config
// - unique_id: {endpoint_type}_{MAC} or {endpoint_type}_{MAC}_{index}
// - name: {dev_name}_{MAC} or {dev_name}_{MAC}_{index}

TEST_P(HAConfigTest, ConfigTopicFollowsPattern) {
    auto params = GetParam();
    
    client.clearPublished();
    send_config();
    
    auto published = client.getPublishedMessages();
    
    // Pattern: homeassistant/{device_type}/{endpoint_type}_{MAC}/config
    // or: homeassistant/{device_type}/{endpoint_type}_{MAC}_{index}/config
    String basePattern = buildBasePattern();
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        // Config topic must start with the base pattern
        EXPECT_GE(msg.topic.indexOf(basePattern), 0)
            << params.config_name << ": Config topic '" << msg.topic.c_str()
            << "' should start with pattern '" << basePattern.c_str() << "'";
        
        // Config topic must NOT contain "_relay_"
        EXPECT_EQ(msg.topic.indexOf("_relay_"), -1)
            << params.config_name << ": Config topic '" << msg.topic.c_str()
            << "' should NOT contain '_relay_'";
        
        // Config topic must end with "/config"
        EXPECT_TRUE(stringEndsWith(msg.topic, "/config"))
            << params.config_name << ": Config topic '" << msg.topic.c_str()
            << "' should end with '/config'";
    }
}

TEST_P(HAConfigTest, UniqueIdFollowsPattern) {
    auto params = GetParam();
    
    client.clearPublished();
    send_config();
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    // Pattern: {endpoint_type}_{MAC} or {endpoint_type}_{MAC}_{index}
    String basePattern = String(params.expected_endpoint_type) + "_" + mac;
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String uniqueId = extractJsonValue(payload, "unique_id");
        
        if (uniqueId.length() == 0) continue;
        
        // unique_id must start with the base pattern
        EXPECT_GE(uniqueId.indexOf(basePattern), 0)
            << params.config_name << ": unique_id '" << uniqueId.c_str()
            << "' should contain pattern '" << basePattern.c_str() << "'";
        
        // unique_id must NOT contain keywords like "_relay_", "_switch_", etc.
        EXPECT_EQ(uniqueId.indexOf("_relay_"), -1)
            << params.config_name << ": unique_id '" << uniqueId.c_str()
            << "' should NOT contain '_relay_'";
        
        EXPECT_EQ(uniqueId.indexOf("_switch_"), -1)
            << params.config_name << ": unique_id '" << uniqueId.c_str()
            << "' should NOT contain '_switch_'";
        
        // unique_id should only contain alphanumeric characters and underscores
        for (unsigned int i = 0; i < uniqueId.length(); i++) {
            EXPECT_TRUE(isValidIdentifierChar(uniqueId[i]))
                << params.config_name << ": unique_id '" << uniqueId.c_str()
                << "' contains invalid character '" << uniqueId[i] << "' at position " << i;
        }
    }
}

TEST_P(HAConfigTest, NameFollowsPattern) {
    auto params = GetParam();
    
    client.clearPublished();
    send_config();
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    // Pattern: {dev_name}_{MAC} or {dev_name}_{MAC}_{index}
    // dev_name is variable, but MAC should be present
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String name = extractJsonValue(payload, "name");
        
        if (name.length() == 0) continue;
        
        // name must contain MAC
        EXPECT_GE(name.indexOf(mac), 0)
            << params.config_name << ": name '" << name.c_str()
            << "' should contain MAC '" << mac.c_str() << "'";
        
        // name must NOT contain keywords like "_relay_", "_switch_", etc.
        EXPECT_EQ(name.indexOf("_relay_"), -1)
            << params.config_name << ": name '" << name.c_str()
            << "' should NOT contain '_relay_'";
        
        EXPECT_EQ(name.indexOf("_switch_"), -1)
            << params.config_name << ": name '" << name.c_str()
            << "' should NOT contain '_switch_'";
    }
}

TEST_P(HAConfigTest, CommandAndStateTopicsFollowPattern) {
    auto params = GetParam();
    
    client.clearPublished();
    send_config();
    
    auto published = client.getPublishedMessages();
    
    // Pattern for command_topic: {device_topic}/cmd/...
    // Pattern for state_topic: {device_topic}/state/...
    String expectedDeviceTopic = buildExpectedDeviceTopic();
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String cmdTopic = extractJsonValue(payload, "command_topic");
        String stateTopic = extractJsonValue(payload, "state_topic");
        
        if (cmdTopic.length() > 0) {
            EXPECT_GE(cmdTopic.indexOf(expectedDeviceTopic), 0)
                << params.config_name << ": command_topic '" << cmdTopic.c_str()
                << "' should start with '" << expectedDeviceTopic.c_str() << "'";
            
            EXPECT_GE(cmdTopic.indexOf("/cmd"), 0)
                << params.config_name << ": command_topic '" << cmdTopic.c_str()
                << "' should contain '/cmd'";
        }
        
        if (stateTopic.length() > 0) {
            EXPECT_GE(stateTopic.indexOf(expectedDeviceTopic), 0)
                << params.config_name << ": state_topic '" << stateTopic.c_str()
                << "' should start with '" << expectedDeviceTopic.c_str() << "'";
            
            EXPECT_GE(stateTopic.indexOf("/state"), 0)
                << params.config_name << ": state_topic '" << stateTopic.c_str()
                << "' should contain '/state'";
        }
    }
}
