/**
 * Device Info Validation Tests
 * 
 * Tests for validating device info structure in MQTT auto-discovery payloads.
 * Validates identifiers, connections, manufacturer, model, and versions.
 */

#include "test_ha_common.h"

// ============================================================================
// Device Info Test Fixture
// ============================================================================

class HADeviceInfoTest : public HAIntegrationTestBase {};

// ============================================================================
// Device Info Validation (per HA schemas.py MQTT_ENTITY_DEVICE_INFO_SCHEMA)
// ============================================================================

TEST_F(HADeviceInfoTest, DeviceInfoHasRequiredIdentifierOrConnection) {
    // Device must have at least one identifying value in 'identifiers' and/or 'connections'
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    EXPECT_TRUE(payload.indexOf("\"device\"") >= 0) 
        << "Light config should include 'device' object";
    
    bool hasIdentifiers = payload.indexOf("\"identifiers\"") >= 0;
    bool hasConnections = payload.indexOf("\"connections\"") >= 0;
    
    EXPECT_TRUE(hasIdentifiers || hasConnections) 
        << "Device must have at least one of 'identifiers' or 'connections'";
}

TEST_F(HADeviceInfoTest, DeviceConnectionsArrayFormat) {
    // connections must be array of [type, identifier] pairs
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"connections\"") >= 0) {
        String connections = extractJsonValue(payload, "connections");
        
        // Should be an array
        EXPECT_EQ(connections[0], '[') << "connections should be an array";
        
        // Should contain connection type like "mac" or "ip"
        bool hasMac = connections.indexOf("\"mac\"") >= 0;
        bool hasIp = connections.indexOf("\"ip\"") >= 0;
        EXPECT_TRUE(hasMac || hasIp) 
            << "connections should include 'mac' or 'ip' connection type";
    }
}

TEST_F(HADeviceInfoTest, DeviceInfoHasValidManufacturer) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"manufacturer\"") >= 0) {
        String manufacturer = extractJsonValue(payload, "manufacturer");
        EXPECT_GT(manufacturer.length(), 0u) << "manufacturer should not be empty if specified";
        EXPECT_LT(manufacturer.length(), 256u) << "manufacturer should be reasonable length";
    }
}

TEST_F(HADeviceInfoTest, DeviceInfoHasValidModel) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"model\"") >= 0) {
        String model = extractJsonValue(payload, "model");
        EXPECT_GT(model.length(), 0u) << "model should not be empty if specified";
        EXPECT_LT(model.length(), 256u) << "model should be reasonable length";
    }
}

TEST_F(HADeviceInfoTest, DeviceInfoVersionsAreStrings) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    // hw_version and sw_version should be strings
    if (payload.indexOf("\"hw_version\"") >= 0) {
        int pos = payload.indexOf("\"hw_version\"");
        int colonPos = indexOfFrom(payload, ':', pos);
        if (colonPos >= 0) {
            int afterColon = colonPos + 1;
            while (afterColon < (int)payload.length() && payload[afterColon] == ' ') afterColon++;
            if (afterColon < (int)payload.length()) {
                EXPECT_EQ(payload[afterColon], '"') << "hw_version should be a string";
            }
        }
    }
    
    if (payload.indexOf("\"sw_version\"") >= 0) {
        int pos = payload.indexOf("\"sw_version\"");
        int colonPos = indexOfFrom(payload, ':', pos);
        if (colonPos >= 0) {
            int afterColon = colonPos + 1;
            while (afterColon < (int)payload.length() && payload[afterColon] == ' ') afterColon++;
            if (afterColon < (int)payload.length()) {
                EXPECT_EQ(payload[afterColon], '"') << "sw_version should be a string";
            }
        }
    }
}

// ============================================================================
// Cross-Entity Device Info Consistency Tests
// ============================================================================

TEST_F(HADeviceInfoTest, LightAndSensorShareSameDeviceIdentifiers) {
    send_config();
    send_sensor_config(0);
    
    String lightPayload = getPublishedPayload(getLightConfigTopic());
    String sensorPayload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    String lightIdentifiers = extractJsonValue(lightPayload, "identifiers");
    String sensorIdentifiers = extractJsonValue(sensorPayload, "identifiers");
    
    if (lightIdentifiers.length() > 0 && sensorIdentifiers.length() > 0) {
        EXPECT_EQ(lightIdentifiers, sensorIdentifiers) 
            << "Light and sensor entities should share the same device identifiers";
    }
}

TEST_F(HADeviceInfoTest, LightAndSensorShareSameManufacturer) {
    send_config();
    send_sensor_config(0);
    
    String lightPayload = getPublishedPayload(getLightConfigTopic());
    String sensorPayload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    String lightManufacturer = extractJsonValue(lightPayload, "manufacturer");
    String sensorManufacturer = extractJsonValue(sensorPayload, "manufacturer");
    
    if (lightManufacturer.length() > 0 && sensorManufacturer.length() > 0) {
        EXPECT_EQ(lightManufacturer, sensorManufacturer) 
            << "Light and sensor entities should share the same manufacturer";
    }
}
