/**
 * Light Entity Auto-Discovery Tests
 * 
 * Tests for MQTT auto-discovery of light entities in Home Assistant.
 * Validates light config topic structure, required fields, and payloads.
 */

#include "test_ha_common.h"

// ============================================================================
// Light Entity Test Fixture
// ============================================================================

class HALightEntityTest : public HAIntegrationTestBase {};

// ============================================================================
// Light Entity Auto-Discovery Tests
// ============================================================================

TEST_F(HALightEntityTest, LightConfigTopicStructure) {
    send_config();
    
    // Verify config topic follows HA convention
    // Format: homeassistant/light/lamp_<MAC>/config
    String expectedTopic = getLightConfigTopic();
    EXPECT_TRUE(wasTopicPublished(expectedTopic));
}

TEST_F(HALightEntityTest, LightConfigContainsUniqueID) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, "\"unique_id\""));
    EXPECT_TRUE(payloadContains(configTopic, (String("lamp_") + getFormattedMAC()).c_str()));
}

TEST_F(HALightEntityTest, LightConfigContainsSchema) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, "\"schema\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"basic\""));
}

TEST_F(HALightEntityTest, LightConfigContainsCommandTopic) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, "\"command_topic\""));
    EXPECT_TRUE(payloadContains(configTopic, "/cmd/lamp_power"));
}

TEST_F(HALightEntityTest, LightConfigContainsStateTopic) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, "\"state_topic\""));
    EXPECT_TRUE(payloadContains(configTopic, "/state/lamp/power"));
}

TEST_F(HALightEntityTest, LightConfigContainsPayloads) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, "\"payload_on\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"payload_off\""));
}

TEST_F(HALightEntityTest, LightConfigContainsSupportedColorModes) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, "\"supported_color_modes\""));
    
#if defined(BRI_CONTROL)
    EXPECT_TRUE(payloadContains(configTopic, "\"brightness\""));
#endif
}

TEST_F(HALightEntityTest, LightConfigContainsDeviceInfo) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    
    // Verify device section
    EXPECT_TRUE(payloadContains(configTopic, "\"device\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"model\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"hw_version\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"sw_version\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"manufacturer\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"identifiers\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"connections\""));
}

TEST_F(HALightEntityTest, LightConfigContainsMACAddress) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, test_mac.c_str()));
}

TEST_F(HALightEntityTest, LightConfigContainsIPAddress) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    EXPECT_TRUE(payloadContains(configTopic, "192.168.4.139"));
}

TEST_F(HALightEntityTest, LightConfigBrightnessTopics) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    
#if defined(BRI_CONTROL)
    EXPECT_TRUE(payloadContains(configTopic, "\"brightness_command_topic\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"brightness_state_topic\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"brightness_scale\""));
    EXPECT_TRUE(payloadContains(configTopic, "/cmd/lamp"));
    EXPECT_TRUE(payloadContains(configTopic, "/state/lamp/prc"));
#endif
}

TEST_F(HALightEntityTest, LightConfigColorTempTopics) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    
#ifdef CCT_CONTROL
    EXPECT_TRUE(payloadContains(configTopic, "\"color_temp_command_topic\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"color_temp_state_topic\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"min_mireds\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"max_mireds\""));
    EXPECT_TRUE(payloadContains(configTopic, "/cmd/cct_mired"));
    EXPECT_TRUE(payloadContains(configTopic, "/state/lamp/cct_mired"));
#endif
}

// ============================================================================
// Light Payload Validation Tests (per HA schema_basic.py)
// ============================================================================

TEST_F(HALightEntityTest, LightPayloadHasRequiredCommandTopic) {
    // command_topic is REQUIRED for MQTT light (MQTT_RW_SCHEMA)
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    EXPECT_TRUE(payload.indexOf("\"command_topic\"") >= 0) 
        << "Light config must include 'command_topic' (required by HA)";
}

TEST_F(HALightEntityTest, LightPayloadHasValidName) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    EXPECT_TRUE(payload.indexOf("\"name\"") >= 0) 
        << "Light config should include 'name' field";
    
    String name = extractJsonValue(payload, "name");
    EXPECT_GT(name.length(), 0u) << "Name should not be empty";
}

TEST_F(HALightEntityTest, LightPayloadHasValidUniqueId) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    EXPECT_TRUE(payload.indexOf("\"unique_id\"") >= 0) 
        << "Light config must include 'unique_id' for entity registry";
    
    String uniqueId = extractJsonValue(payload, "unique_id");
    EXPECT_GT(uniqueId.length(), 0u) << "unique_id should not be empty";
}

TEST_F(HALightEntityTest, LightPayloadHasValidPayloadOn) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"payload_on\"") >= 0) {
        String payloadOn = extractJsonValue(payload, "payload_on");
        EXPECT_GT(payloadOn.length(), 0u) << "payload_on should not be empty if specified";
    }
}

TEST_F(HALightEntityTest, LightPayloadHasValidPayloadOff) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"payload_off\"") >= 0) {
        String payloadOff = extractJsonValue(payload, "payload_off");
        EXPECT_GT(payloadOff.length(), 0u) << "payload_off should not be empty if specified";
    }
}

TEST_F(HALightEntityTest, LightSupportedColorModesAreValid) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"supported_color_modes\"") >= 0) {
        String colorModes = extractJsonValue(payload, "supported_color_modes");
        
        // Valid HA ColorMode values (from homeassistant/components/light/const.py)
        const char* validModes[] = {
            "unknown", "onoff", "brightness", "color_temp", 
            "hs", "xy", "rgb", "rgbw", "rgbww", "white"
        };
        
        EXPECT_TRUE(colorModes.indexOf("invalid") < 0) 
            << "supported_color_modes should not contain 'invalid'";
        EXPECT_TRUE(colorModes.indexOf("unknown_mode") < 0) 
            << "supported_color_modes should not contain 'unknown_mode'";
        
        bool hasAtLeastOneValidMode = false;
        for (size_t i = 0; i < sizeof(validModes) / sizeof(validModes[0]); i++) {
            if (colorModes.indexOf(validModes[i]) >= 0) {
                hasAtLeastOneValidMode = true;
                break;
            }
        }
        
        if (colorModes.length() > 2) {
            EXPECT_TRUE(hasAtLeastOneValidMode)
                << "supported_color_modes should contain at least one valid mode";
        }
    }
}

TEST_F(HALightEntityTest, LightBrightnessScaleIsPositiveInteger) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"brightness_scale\"") >= 0) {
        String scaleStr = extractJsonValue(payload, "brightness_scale");
        int scale = atoi(scaleStr.c_str());
        EXPECT_GE(scale, 1) << "brightness_scale must be >= 1";
        EXPECT_LE(scale, 1000) << "brightness_scale should be reasonable (<=1000)";
    }
}

TEST_F(HALightEntityTest, LightColorTempMiredsAreValid) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"min_mireds\"") >= 0) {
        String minMiredsStr = extractJsonValue(payload, "min_mireds");
        int minMireds = atoi(minMiredsStr.c_str());
        
        EXPECT_GT(minMireds, 0) << "min_mireds must be positive";
        EXPECT_GE(minMireds, 100) << "min_mireds typically >= 100 (10000K)";
        EXPECT_LE(minMireds, 600) << "min_mireds typically <= 600 (~1667K)";
    }
    
    if (payload.indexOf("\"max_mireds\"") >= 0) {
        String maxMiredsStr = extractJsonValue(payload, "max_mireds");
        int maxMireds = atoi(maxMiredsStr.c_str());
        
        EXPECT_GT(maxMireds, 0) << "max_mireds must be positive";
        
        if (payload.indexOf("\"min_mireds\"") >= 0) {
            String minMiredsStr = extractJsonValue(payload, "min_mireds");
            int minMireds = atoi(minMiredsStr.c_str());
            EXPECT_GT(maxMireds, minMireds) << "max_mireds should be > min_mireds";
        }
    }
}

TEST_F(HALightEntityTest, LightSchemaIsBasicWhenSpecified) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    if (payload.indexOf("\"schema\"") >= 0) {
        String schema = extractJsonValue(payload, "schema");
        EXPECT_TRUE(schema == "basic" || schema == "json" || schema == "template")
            << "schema must be 'basic', 'json', or 'template', got: " << schema.c_str();
    }
}

TEST_F(HALightEntityTest, LightPayloadIsValidJSONWithMatchingBraces) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    int curlyOpen = 0, curlyClose = 0;
    int squareOpen = 0, squareClose = 0;
    
    for (unsigned int i = 0; i < payload.length(); i++) {
        switch (payload[i]) {
            case '{': curlyOpen++; break;
            case '}': curlyClose++; break;
            case '[': squareOpen++; break;
            case ']': squareClose++; break;
        }
    }
    
    EXPECT_EQ(curlyOpen, curlyClose) << "Curly braces must match";
    EXPECT_EQ(squareOpen, squareClose) << "Square brackets must match";
    EXPECT_GT(curlyOpen, 0) << "JSON should have at least one object";
}

TEST_F(HALightEntityTest, LightPayloadDoesNotHaveTrailingComma) {
    send_config();
    String payload = getPublishedPayload(getLightConfigTopic());
    
    EXPECT_TRUE(payload.indexOf(",}") < 0) << "JSON should not have trailing comma before }";
    EXPECT_TRUE(payload.indexOf(",]") < 0) << "JSON should not have trailing comma before ]";
}

TEST_F(HALightEntityTest, ConfigPayloadNotEmpty) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    String payload = getPublishedPayload(configTopic);
    
    EXPECT_GT(payload.length(), 0);
    EXPECT_GT(payload.length(), 100);
}

TEST_F(HALightEntityTest, ConfigPayloadSizeReasonable) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    String payload = getPublishedPayload(configTopic);
    
    EXPECT_LT(payload.length(), 2048);
}

TEST_F(HALightEntityTest, ConfigContainsValidJSON) {
    send_config();
    
    String configTopic = getLightConfigTopic();
    String payload = getPublishedPayload(configTopic);
    
    int open_braces = 0;
    int close_braces = 0;
    for (unsigned int i = 0; i < payload.length(); i++) {
        if (payload[i] == '{') open_braces++;
        if (payload[i] == '}') close_braces++;
    }
    EXPECT_EQ(open_braces, close_braces);
    EXPECT_GT(open_braces, 0);
}
