/**
 * Online Status and Integration Tests
 * 
 * Tests for online status triggers, topic structure, naming conventions,
 * edge cases, and boundary conditions.
 */

#include "test_ha_common.h"

// ============================================================================
// Online Status Test Fixture
// ============================================================================

class HAOnlineStatusTest : public HAIntegrationTestBase {};

// ============================================================================
// Online Status Integration Tests
// ============================================================================

TEST_F(HAOnlineStatusTest, OnlineStatusTriggersLightDiscovery) {
    // Simulate receiving "online" status message
    const char* topic = "homeassistant/status";
    const char* payload = "online";
    
    callback((char*)topic, (byte*)payload, strlen(payload));
    
    // Verify light config was published
    String lightConfigTopic = getLightConfigTopic();
    EXPECT_TRUE(wasTopicPublished(lightConfigTopic));
}

TEST_F(HAOnlineStatusTest, OnlineStatusTriggersSensorDiscovery) {
    // Simulate receiving "online" status message
    const char* topic = "homeassistant/status";
    const char* payload = "online";
    
    callback((char*)topic, (byte*)payload, strlen(payload));
    
    // Verify sensor config was published
    String sensorConfigTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(wasTopicPublished(sensorConfigTopic));
}

TEST_F(HAOnlineStatusTest, OnlineStatusPublishesCurrentState) {
    // Set some state
    set_power(true);
    set_out_prc(75);
    
    client.clearPublished();
    
    // Simulate "online" status
    const char* topic = "homeassistant/status";
    const char* payload = "online";
    callback((char*)topic, (byte*)payload, strlen(payload));
    
    // Verify state was published
    auto published = client.getPublishedMessages();
    bool found_state = false;
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/state/") >= 0) {
            found_state = true;
            break;
        }
    }
    EXPECT_TRUE(found_state);
}

TEST_F(HAOnlineStatusTest, OnlineStatusIgnoredWhenOffline) {
    client.setConnected(false);
    
    const char* topic = "homeassistant/status";
    const char* payload = "online";
    
    callback((char*)topic, (byte*)payload, strlen(payload));
    
    auto published = client.getPublishedMessages();
    
    // Should not publish anything when MQTT is disconnected
    EXPECT_EQ(published.size(), 0);
}

TEST_F(HAOnlineStatusTest, NonOnlineStatusIgnored) {
    const char* topic = "homeassistant/status";
    const char* payload = "offline";
    
    callback((char*)topic, (byte*)payload, strlen(payload));
    
    auto published = client.getPublishedMessages();
    
    // Should not trigger discovery for non-"online" messages
    EXPECT_EQ(published.size(), 0);
}

// ============================================================================
// Topic Structure and Naming Tests
// ============================================================================

class HATopicStructureTest : public HAIntegrationTestBase {};

TEST_F(HATopicStructureTest, AllTopicsContainMAC) {
    send_config();
    send_sensor_config();
    
    auto published = client.getPublishedMessages();
    String formattedMAC = getFormattedMAC();
    
    // All discovery topics should contain the MAC address
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") >= 0) {
            EXPECT_TRUE(msg.topic.indexOf(formattedMAC.c_str()) >= 0);
        }
    }
}

TEST_F(HATopicStructureTest, AllTopicsStartWithPrefix) {
    send_config();
    send_sensor_config();
    
    auto published = client.getPublishedMessages();
    
    // All topics should start with the configured prefix
    for (const auto& msg : published) {
        EXPECT_TRUE(msg.topic.startsWith(topic_prefix));
    }
}

TEST_F(HATopicStructureTest, LightTopicsContainDeviceType) {
    send_config();
    
    auto published = client.getPublishedMessages();
    
    // Light topics should contain device_type
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/light/") >= 0) {
            EXPECT_TRUE(msg.topic.indexOf("lamp") >= 0);
        }
    }
}

TEST_F(HATopicStructureTest, SensorTopicsContainSensorPrefix) {
    send_sensor_config();
    
    auto published = client.getPublishedMessages();
    
    // Sensor topics should contain /sensor/
    for (const auto& msg : published) {
        if (msg.topic.indexOf("_RSSI") >= 0) {
            EXPECT_TRUE(msg.topic.indexOf("/sensor/") >= 0);
        }
    }
}

// ============================================================================
// Edge Cases and Boundary Conditions
// ============================================================================

class HAEdgeCasesTest : public HAIntegrationTestBase {};

TEST_F(HAEdgeCasesTest, ConfigNotPublishedWhenMQTTDisconnected) {
    client.setConnected(false);
    
    send_config();
    
    auto published = client.getPublishedMessages();
    EXPECT_EQ(published.size(), 0);
}

TEST_F(HAEdgeCasesTest, SensorConfigNotPublishedWhenMQTTDisconnected) {
    client.setConnected(false);
    
    send_sensor_config();
    
    auto published = client.getPublishedMessages();
    EXPECT_EQ(published.size(), 0);
}

TEST_F(HAEdgeCasesTest, RepeatedConfigCallsPublishMultipleTimes) {
    send_config();
    send_config();
    send_config();
    
    auto published = client.getPublishedMessages();
    
    // Should publish each time it's called
    int config_count = 0;
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") >= 0 && msg.topic.indexOf("/light/") >= 0) {
            config_count++;
        }
    }
    EXPECT_EQ(config_count, 3);
}

TEST_F(HAEdgeCasesTest, MQTTBufferResetAfterPublish) {
    send_config();
    
    // Buffer should be reset to default size after publishing
    EXPECT_EQ(client.getBufferSize(), MQTT_MAX_PACKET_SIZE);
}
