/**
 * Sensor Entity Auto-Discovery Tests
 * 
 * Tests for MQTT auto-discovery of sensor entities in Home Assistant.
 * Validates sensor config topic structure, required fields, and payloads.
 */

#include "test_ha_common.h"

// ============================================================================
// Sensor Entity Test Fixture
// ============================================================================

class HASensorEntityTest : public HAIntegrationTestBase {};

// ============================================================================
// Sensor Entity Auto-Discovery Tests
// ============================================================================

TEST_F(HASensorEntityTest, SensorConfigTopicStructure) {
    send_sensor_config(0);
    
    // Verify sensor config topic follows HA convention
    // Format: homeassistant/sensor/lamp_<MAC>_RSSI/config
    String expectedTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(wasTopicPublished(expectedTopic));
}

TEST_F(HASensorEntityTest, SensorConfigContainsUniqueID) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"unique_id\""));
    EXPECT_TRUE(payloadContains(configTopic, "sensor_RSSI"));
}

TEST_F(HASensorEntityTest, SensorConfigContainsDeviceClass) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"device_class\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"signal_strength\""));
}

TEST_F(HASensorEntityTest, SensorConfigContainsStateClass) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"state_class\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"measurement\""));
}

TEST_F(HASensorEntityTest, SensorConfigContainsUnitOfMeasurement) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"unit_of_measurement\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"dBm\""));
}

TEST_F(HASensorEntityTest, SensorConfigContainsEntityCategory) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"entity_category\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"diagnostic\""));
}

TEST_F(HASensorEntityTest, SensorConfigContainsEnabledByDefault) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"enabled_by_default\""));
    EXPECT_TRUE(payloadContains(configTopic, "false"));
}

TEST_F(HASensorEntityTest, SensorConfigContainsValueTemplate) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"value_template\""));
    EXPECT_TRUE(payloadContains(configTopic, "{{ value_json.RSSI }}"));
}

TEST_F(HASensorEntityTest, SensorConfigContainsStateTopic) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    EXPECT_TRUE(payloadContains(configTopic, "\"state_topic\""));
    EXPECT_TRUE(payloadContains(configTopic, "/sensor/"));
    EXPECT_TRUE(payloadContains(configTopic, "/state"));
}

TEST_F(HASensorEntityTest, SensorConfigContainsDeviceInfo) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    
    EXPECT_TRUE(payloadContains(configTopic, "\"device\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"model\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"manufacturer\""));
    EXPECT_TRUE(payloadContains(configTopic, "\"identifiers\""));
    EXPECT_TRUE(payloadContains(configTopic, getFormattedMAC().c_str()));
}

TEST_F(HASensorEntityTest, MultipleSensorsPublished) {
    send_sensor_config();
    
    auto published = client.getPublishedMessages();
    
    int sensor_configs = 0;
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/sensor/") >= 0 && msg.topic.indexOf("/config") >= 0) {
            sensor_configs++;
        }
    }
    
    EXPECT_EQ(sensor_configs, sensors_count);
}

// ============================================================================
// Sensor Payload Validation Tests (per HA sensor.py and sensor/const.py)
// ============================================================================

TEST_F(HASensorEntityTest, SensorPayloadHasRequiredStateTopic) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    EXPECT_TRUE(payload.indexOf("\"state_topic\"") >= 0) 
        << "Sensor config must include 'state_topic' (required by HA)";
}

TEST_F(HASensorEntityTest, SensorDeviceClassIsValid) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    if (payload.indexOf("\"device_class\"") >= 0) {
        String deviceClass = extractJsonValue(payload, "device_class");
        
        const char* validClasses[] = {
            "apparent_power", "aqi", "atmospheric_pressure", "battery",
            "carbon_dioxide", "carbon_monoxide", "current", "data_rate",
            "data_size", "date", "distance", "duration", "energy",
            "energy_storage", "enum", "frequency", "gas", "humidity",
            "illuminance", "irradiance", "moisture", "monetary",
            "nitrogen_dioxide", "nitrogen_monoxide", "nitrous_oxide", "ozone",
            "ph", "pm1", "pm10", "pm25", "power", "power_factor",
            "precipitation", "precipitation_intensity", "pressure",
            "reactive_power", "signal_strength", "sound_pressure", "speed",
            "sulphur_dioxide", "temperature", "timestamp",
            "volatile_organic_compounds", "voltage", "volume",
            "volume_flow_rate", "water", "weight", "wind_speed"
        };
        
        bool isValid = false;
        for (size_t i = 0; i < sizeof(validClasses) / sizeof(validClasses[0]); i++) {
            if (deviceClass == validClasses[i]) {
                isValid = true;
                break;
            }
        }
        EXPECT_TRUE(isValid) 
            << "device_class '" << deviceClass.c_str() << "' must be a valid SensorDeviceClass";
    }
}

TEST_F(HASensorEntityTest, SensorStateClassIsValid) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    if (payload.indexOf("\"state_class\"") >= 0) {
        String stateClass = extractJsonValue(payload, "state_class");
        
        const char* validStateClasses[] = {
            "measurement", "measurement_angle", "total", "total_increasing"
        };
        
        bool isValid = false;
        for (size_t i = 0; i < sizeof(validStateClasses) / sizeof(validStateClasses[0]); i++) {
            if (stateClass == validStateClasses[i]) {
                isValid = true;
                break;
            }
        }
        EXPECT_TRUE(isValid) 
            << "state_class '" << stateClass.c_str() << "' must be valid SensorStateClass";
    }
}

TEST_F(HASensorEntityTest, SensorEntityCategoryIsValid) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    if (payload.indexOf("\"entity_category\"") >= 0) {
        String entityCategory = extractJsonValue(payload, "entity_category");
        
        EXPECT_TRUE(entityCategory == "diagnostic" || entityCategory == "config")
            << "entity_category must be 'diagnostic' or 'config', got: " << entityCategory.c_str();
    }
}

TEST_F(HASensorEntityTest, SensorEnabledByDefaultIsBoolean) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    if (payload.indexOf("\"enabled_by_default\"") >= 0) {
        String enabledByDefault = extractJsonValue(payload, "enabled_by_default");
        EXPECT_TRUE(enabledByDefault == "true" || enabledByDefault == "false")
            << "enabled_by_default must be boolean, got: " << enabledByDefault.c_str();
    }
}

TEST_F(HASensorEntityTest, SensorValueTemplateFormat) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    if (payload.indexOf("\"value_template\"") >= 0) {
        String valueTemplate = extractJsonValue(payload, "value_template");
        
        EXPECT_TRUE(valueTemplate.indexOf("{{") >= 0) 
            << "value_template should contain Jinja2 opening braces {{";
        EXPECT_TRUE(valueTemplate.indexOf("}}") >= 0) 
            << "value_template should contain Jinja2 closing braces }}";
    }
}

TEST_F(HASensorEntityTest, SensorUnitOfMeasurementMatchesDeviceClass) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    String deviceClass = extractJsonValue(payload, "device_class");
    String unit = extractJsonValue(payload, "unit_of_measurement");
    
    if (deviceClass == "signal_strength") {
        EXPECT_TRUE(unit == "dB" || unit == "dBm")
            << "signal_strength device class requires unit 'dB' or 'dBm', got: " << unit.c_str();
    }
}

TEST_F(HASensorEntityTest, SensorPayloadIsValidJSONWithMatchingBraces) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    int curlyOpen = 0, curlyClose = 0;
    int squareOpen = 0, squareClose = 0;
    
    for (unsigned int i = 0; i < payload.length(); i++) {
        switch (payload[i]) {
            case '{': curlyOpen++; break;
            case '}': curlyClose++; break;
            case '[': squareOpen++; break;
            case ']': squareClose++; break;
        }
    }
    
    EXPECT_EQ(curlyOpen, curlyClose) << "Curly braces must match";
    EXPECT_EQ(squareOpen, squareClose) << "Square brackets must match";
    EXPECT_GT(curlyOpen, 0) << "JSON should have at least one object";
}

TEST_F(HASensorEntityTest, SensorPayloadDoesNotHaveTrailingComma) {
    send_sensor_config(0);
    String payload = getPublishedPayload(getSensorConfigTopic("RSSI"));
    
    EXPECT_TRUE(payload.indexOf(",}") < 0) << "JSON should not have trailing comma before }";
    EXPECT_TRUE(payload.indexOf(",]") < 0) << "JSON should not have trailing comma before ]";
}

TEST_F(HASensorEntityTest, SensorConfigPayloadNotEmpty) {
    send_sensor_config(0);
    
    String configTopic = getSensorConfigTopic("RSSI");
    String payload = getPublishedPayload(configTopic);
    
    EXPECT_GT(payload.length(), 0);
    EXPECT_GT(payload.length(), 50);
}
