/**
 * Topic Consistency Tests
 * 
 * Tests that verify the topics and values declared in auto-discovery
 * match the topics actually used by the code for publishing/subscribing.
 * 
 * This ensures that:
 * 1. Topics in auto-discovery config match topics used for publishing state
 * 2. Command topics in auto-discovery match topics the callback expects
 * 3. Payload types (on/off values) match what the code actually sends/receives
 * 4. Value ranges (brightness_scale, mireds) match the actual code behavior
 */

#include "test_ha_common.h"

// ============================================================================
// Topic Consistency Test Fixture
// ============================================================================

class HATopicConsistencyTest : public HAIntegrationTestBase {
protected:
    // Extract a string value from JSON payload
    String extractStringValue(const String& payload, const String& key) {
        return extractJsonValue(payload, key);
    }
    
    // Extract an integer value from JSON payload
    int extractIntValue(const String& payload, const String& key) {
        String value = extractJsonValue(payload, key);
        return atoi(value.c_str());
    }
    
    // Helper to get config payload
    String getConfigPayload() {
        send_config();
        return getPublishedPayload(getLightConfigTopic());
    }
    
    // Helper to get sensor config payload
    String getSensorConfigPayloadForIndex(int index) {
        send_sensor_config(index);
        return getPublishedPayload(getSensorConfigTopic(sensors_conf[index].default_name));
    }
    
    // Helper to clear all messages and trigger new publishes
    void resetAndPublish() {
        client.clearPublished();
    }
};

// ============================================================================
// Light Command Topic Consistency Tests
// ============================================================================

TEST_F(HATopicConsistencyTest, PowerCommandTopicMatchesCallbackHandler) {
    // Get the command topic from auto-discovery
    String configPayload = getConfigPayload();
    String commandTopic = extractStringValue(configPayload, "command_topic");
    
    // The command topic should be what the callback expects
    // In callback: subTopic == endpoint_type + "_power"
    String expectedSuffix = String(endpoint_type) + "_power";
    
    EXPECT_TRUE(commandTopic.indexOf(expectedSuffix) >= 0)
        << "command_topic '" << commandTopic.c_str() 
        << "' should contain '" << expectedSuffix.c_str() << "'";
}

TEST_F(HATopicConsistencyTest, PowerStateTopicMatchesPublishFunction) {
    // Get state topic from auto-discovery config
    String configPayload = getConfigPayload();
    String stateTopic = extractStringValue(configPayload, "state_topic");
    
    // Simulate power state publish
    resetAndPublish();
    set_power(true);
    
    // Check what topic was actually used to publish
    auto published = client.getPublishedMessages();
    bool found = false;
    String expectedPattern = String(endpoint_type) + "/power";
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/power") >= 0) {
            // The actual publish topic should match what's in config
            EXPECT_TRUE(msg.topic.indexOf(expectedPattern) >= 0)
                << "Actual power state topic should contain endpoint_type/power pattern";
            found = true;
            break;
        }
    }
    EXPECT_TRUE(found) << "Power state should have been published";
}

#if defined(BRI_CONTROL)
TEST_F(HATopicConsistencyTest, BrightnessCommandTopicMatchesCallbackHandler) {
    // Get brightness command topic from config
    String configPayload = getConfigPayload();
    String briCmdTopic = extractStringValue(configPayload, "brightness_command_topic");
    
    // The brightness command topic should match what callback expects
    // In callback: subTopic.equals(endpoint_type) for brightness
    EXPECT_TRUE(briCmdTopic.indexOf(endpoint_type) >= 0)
        << "brightness_command_topic should contain endpoint_type";
}

TEST_F(HATopicConsistencyTest, BrightnessStateTopicMatchesPublishFunction) {
    // Get brightness state topic from config
    String configPayload = getConfigPayload();
    String briStateTopic = extractStringValue(configPayload, "brightness_state_topic");
    
    // Simulate brightness publish
    resetAndPublish();
    set_out_prc(50);
    
    // Check what was published
    auto published = client.getPublishedMessages();
    bool found = false;
    String expectedPattern = String(endpoint_type) + "/prc";
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/prc") >= 0) {
            // Verify the pattern matches
            EXPECT_TRUE(msg.topic.indexOf(expectedPattern) >= 0)
                << "Actual brightness state topic should match pattern";
            found = true;
            break;
        }
    }
    EXPECT_TRUE(found) << "Brightness state should have been published";
}

TEST_F(HATopicConsistencyTest, BrightnessScaleMatchesActualRange) {
    // Get brightness_scale from config
    String configPayload = getConfigPayload();
    int brightnessScale = extractIntValue(configPayload, "brightness_scale");
    
    // The code uses 0-100 for percentage (in set_out_prc)
    EXPECT_EQ(brightnessScale, 100) 
        << "brightness_scale should be 100 (matching the 0-100% range used in code)";
}
#endif

#ifdef CCT_CONTROL
TEST_F(HATopicConsistencyTest, ColorTempCommandTopicMatchesCallbackHandler) {
    // Get color temp command topic
    String configPayload = getConfigPayload();
    String cctCmdTopic = extractStringValue(configPayload, "color_temp_command_topic");
    
    // In callback: subTopic == "cct_mired"
    EXPECT_TRUE(cctCmdTopic.indexOf("cct_mired") >= 0)
        << "color_temp_command_topic '" << cctCmdTopic.c_str() 
        << "' should contain 'cct_mired'";
}

TEST_F(HATopicConsistencyTest, ColorTempStateTopicMatchesPublishFunction) {
    // Get color temp state topic from config  
    String configPayload = getConfigPayload();
    String cctStateTopic = extractStringValue(configPayload, "color_temp_state_topic");
    
    // Verify it contains cct_mired (that's what send_cct publishes)
    EXPECT_TRUE(cctStateTopic.indexOf("cct_mired") >= 0)
        << "color_temp_state_topic should use cct_mired";
}

TEST_F(HATopicConsistencyTest, MiredsRangeMatchesCodeConstants) {
    // Get mireds from config
    String configPayload = getConfigPayload();
    int minMireds = extractIntValue(configPayload, "min_mireds");
    int maxMireds = extractIntValue(configPayload, "max_mireds");
    
    // These should match CCT_END_MIRED and CCT_START_MIRED from code
    // CCT_START_MIRED = 1000000/CCT_START, CCT_END_MIRED = 1000000/CCT_END
    // min_mireds is the colder temperature (higher Kelvin = lower mired)
    // max_mireds is the warmer temperature (lower Kelvin = higher mired)
    
    EXPECT_EQ(minMireds, CCT_END_MIRED) 
        << "min_mireds should equal CCT_END_MIRED (coldest temp)";
    EXPECT_EQ(maxMireds, CCT_START_MIRED) 
        << "max_mireds should equal CCT_START_MIRED (warmest temp)";
}
#endif

// ============================================================================
// Payload Value Consistency Tests
// ============================================================================

TEST_F(HATopicConsistencyTest, PayloadOnMatchesActualOnValue) {
    // Get payload_on from config
    String configPayload = getConfigPayload();
    String payloadOn = extractStringValue(configPayload, "payload_on");
    
    // In code, set_power(true) and switch_io use 1 for ON
    // Config uses "1" for payload_on
    EXPECT_EQ(payloadOn, "1") 
        << "payload_on should be '1' matching the code's ON value";
}

TEST_F(HATopicConsistencyTest, PayloadOffMatchesActualOffValue) {
    // Get payload_off from config
    String configPayload = getConfigPayload();
    String payloadOff = extractStringValue(configPayload, "payload_off");
    
    // In code, set_power(false) and switch_io use 0 for OFF
    EXPECT_EQ(payloadOff, "0") 
        << "payload_off should be '0' matching the code's OFF value";
}

TEST_F(HATopicConsistencyTest, ActualPowerStatePublishMatchesPayloads) {
    // Turn on and check published value
    resetAndPublish();
    set_power(true);
    
    auto published = client.getPublishedMessages();
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/power") >= 0) {
            // The code publishes "1" for true, "0" for false
            EXPECT_EQ(msg.payload, "1") 
                << "Actual ON state should publish '1'";
            break;
        }
    }
    
    // Turn off and check
    resetAndPublish();
    set_power(false);
    
    published = client.getPublishedMessages();
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/power") >= 0) {
            EXPECT_EQ(msg.payload, "0") 
                << "Actual OFF state should publish '0'";
            break;
        }
    }
}

// ============================================================================
// Sensor Topic Consistency Tests
// ============================================================================

TEST_F(HATopicConsistencyTest, SensorStateTopicMatchesPublishFunction) {
    // Get sensor state topic from config
    String sensorPayload = getSensorConfigPayloadForIndex(0);
    String sensorStateTopic = extractStringValue(sensorPayload, "state_topic");
    
    // The sensor state topic should be device_sensor_topic + "/state"
    // which is what send_all_sensors() uses
    EXPECT_TRUE(sensorStateTopic.indexOf("/sensor/") >= 0)
        << "Sensor state_topic should contain '/sensor/'";
    EXPECT_TRUE(sensorStateTopic.indexOf("/state") >= 0)
        << "Sensor state_topic should end with '/state'";
}

TEST_F(HATopicConsistencyTest, SensorValueTemplateMatchesJSONKey) {
    // Get value_template from sensor config
    String sensorPayload = getSensorConfigPayloadForIndex(0);
    String valueTemplate = extractStringValue(sensorPayload, "value_template");
    
    // The template should reference the sensor's default_name
    // Template format: {{ value_json.SENSOR_NAME }}
    String sensorName = sensors_conf[0].default_name;
    EXPECT_TRUE(valueTemplate.indexOf(sensorName) >= 0)
        << "value_template should contain sensor name";
    EXPECT_TRUE(valueTemplate.indexOf("value_json") >= 0)
        << "value_template should use value_json format";
}

TEST_F(HATopicConsistencyTest, SensorDeviceClassMatchesConfigured) {
    // Verify the device_class in discovery matches sensor_conf
    String sensorPayload = getSensorConfigPayloadForIndex(0);
    String deviceClass = extractStringValue(sensorPayload, "device_class");
    
    EXPECT_EQ(deviceClass, String(sensors_conf[0].device_class))
        << "device_class should match sensors_conf value";
}

TEST_F(HATopicConsistencyTest, SensorStateClassMatchesConfigured) {
    // Verify state_class matches
    String sensorPayload = getSensorConfigPayloadForIndex(0);
    String stateClass = extractStringValue(sensorPayload, "state_class");
    
    EXPECT_EQ(stateClass, String(sensors_conf[0].state_class))
        << "state_class should match sensors_conf value";
}

TEST_F(HATopicConsistencyTest, SensorUnitMatchesConfigured) {
    // Verify unit_of_measurement matches
    String sensorPayload = getSensorConfigPayloadForIndex(0);
    String unit = extractStringValue(sensorPayload, "unit_of_measurement");
    
    EXPECT_EQ(unit, String(sensors_conf[0].unit_of_measurement))
        << "unit_of_measurement should match sensors_conf value";
}

// ============================================================================
// Command Subscription Consistency Tests
// ============================================================================

TEST_F(HATopicConsistencyTest, CallbackHandlesPowerCommandFromDiscoveredTopic) {
    // Get the command topic from discovery
    String configPayload = getConfigPayload();
    String commandTopic = extractStringValue(configPayload, "command_topic");
    
    // Simulate receiving a command on that topic
    // The callback should handle it
    resetAndPublish();
    
    // Construct full topic as callback would receive it
    callback((char*)commandTopic.c_str(), (byte*)"1", 1);
    
    // Verify power was actually turned on
    EXPECT_TRUE(get_power()) << "Power should be ON after command";
    
    // Now send OFF
    callback((char*)commandTopic.c_str(), (byte*)"0", 1);
    EXPECT_FALSE(get_power()) << "Power should be OFF after command";
}

#if defined(BRI_CONTROL)
TEST_F(HATopicConsistencyTest, CallbackHandlesBrightnessFromDiscoveredTopic) {
    // The brightness_command_topic in config is: cmd_topic + "/" + endpoint_type
    // Which equals: device_topic + "/cmd/" + endpoint_type
    // The callback extracts subTopic = topicStr.substring(cmd_topic.length() + 1)
    // and checks subTopic.equals(endpoint_type) for brightness control
    
    // Build the expected brightness command topic
    String briCmdTopic = cmd_topic + "/" + String(endpoint_type);
    
    // Send brightness command of 50%
    callback((char*)briCmdTopic.c_str(), (byte*)"50", 2);
    
    // Verify brightness was set by checking main_state
    // With PWM_CONTROL: on_procents is mapped from 0-100 to 0-PWM_MAX
    // Without PWM: it's mapped from 0-100 to 0-PinsCount
    // send_prc then maps it back to 0-100 for publishing
    // Due to integer division rounding, we may get ±1 difference
    
    // Force a publish to see the value
    resetAndPublish();
    send_prc(main_state.on_procents);
    
    auto published = client.getPublishedMessages();
    bool found = false;
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/prc") >= 0) {
            // Should have published approximately 50 (±1 due to integer rounding)
            int publishedValue = msg.payload.toInt();
            EXPECT_GE(publishedValue, 49) 
                << "Brightness should be approximately 50 after command";
            EXPECT_LE(publishedValue, 51) 
                << "Brightness should be approximately 50 after command";
            found = true;
            break;
        }
    }
    EXPECT_TRUE(found) << "Brightness state should have been published";
}
#endif

#ifdef CCT_CONTROL
TEST_F(HATopicConsistencyTest, CallbackHandlesCCTFromDiscoveredTopic) {
    // Get CCT command topic from discovery
    String configPayload = getConfigPayload();
    String cctCmdTopic = extractStringValue(configPayload, "color_temp_command_topic");
    
    // Calculate a valid mired value in range
    int testMired = (CCT_START_MIRED + CCT_END_MIRED) / 2;
    String miredStr = String(testMired);
    
    // Send CCT command
    callback((char*)cctCmdTopic.c_str(), (byte*)miredStr.c_str(), miredStr.length());
    
    // The command should have been processed (CCT value changed)
    // We can verify by checking that state was updated
    SUCCEED(); // If no crash, command was handled
}
#endif

// ============================================================================
// Online Status Discovery Trigger Tests
// ============================================================================

TEST_F(HATopicConsistencyTest, OnlineStatusTopicMatchesSubscription) {
    // The callback handles topic_prefix + "/status" with payload "online"
    String statusTopic = String(topic_prefix) + "/status";
    
    // This should trigger send_config() and send_sensor_config()
    resetAndPublish();
    callback((char*)statusTopic.c_str(), (byte*)"online", 6);
    
    // Verify configs were published
    auto published = client.getPublishedMessages();
    bool foundLightConfig = false;
    bool foundSensorConfig = false;
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/light/") >= 0 && msg.topic.indexOf("/config") >= 0) {
            foundLightConfig = true;
        }
        if (msg.topic.indexOf("/sensor/") >= 0 && msg.topic.indexOf("/config") >= 0) {
            foundSensorConfig = true;
        }
    }
    
    EXPECT_TRUE(foundLightConfig) << "Light config should be published on 'online' status";
    EXPECT_TRUE(foundSensorConfig) << "Sensor config should be published on 'online' status";
}

// ============================================================================
// Identifier Consistency Tests
// ============================================================================

TEST_F(HATopicConsistencyTest, UniqueIdMatchesTopicNaming) {
    // Get unique_id from config
    String configPayload = getConfigPayload();
    String uniqueId = extractStringValue(configPayload, "unique_id");
    
    // unique_id should contain endpoint_type and MAC
    String mac = getFormattedMAC();
    EXPECT_TRUE(uniqueId.indexOf(endpoint_type) >= 0)
        << "unique_id should contain endpoint_type";
    EXPECT_TRUE(uniqueId.indexOf(mac) >= 0)
        << "unique_id should contain MAC address";
}

TEST_F(HATopicConsistencyTest, DeviceIdentifiersMatchAcrossEntities) {
    // Get identifiers from light and sensor configs
    String lightPayload = getConfigPayload();
    String lightIdentifiers = extractStringValue(lightPayload, "identifiers");
    
    client.clearPublished();
    String sensorPayload = getSensorConfigPayloadForIndex(0);
    String sensorIdentifiers = extractStringValue(sensorPayload, "identifiers");
    
    // Both should use the same identifier (MAC)
    EXPECT_EQ(lightIdentifiers, sensorIdentifiers)
        << "Light and sensor should use same device identifier";
    EXPECT_EQ(lightIdentifiers, getFormattedMAC())
        << "Identifier should be the MAC address";
}
