#ifndef TEST_HELPERS_H
#define TEST_HELPERS_H

#include "Arduino.h"
#include "ESP8266WiFi.h"
#include "PubSubClient.h"
#include "ArduinoJson.h"
#include "FileData.h"
#include "LittleFS.h"
#include "Ticker.h"
#include <map>

// Base structures
#define OFF HIGH
#define ON LOW

struct pin_state_str {
    int pin;
    bool state;
    int pwm;
#ifdef PULSE_CONTROL
    int state_pin;
#endif
};

struct main_state_str {
    unsigned on_procents;
#ifdef CCT_CONTROL
    unsigned led_cct;
#endif
#ifdef WSRGB
    uint8_t rgb_r;
    uint8_t rgb_g;
    uint8_t rgb_b;
#endif
};

struct sensor_state_str {
    const char *default_name;
    const char *device_class;
    const char *state_class;
    const char *unit_of_measurement;
    const bool diag;
    const bool hide;
    void *value;
    void *func;
};

struct vendor_info_str {
    const char *manufacturer;
    const char *model;
    const char *hw_version;
    const char *sw_version;
    const char *support_url;
};

// Mock configuration - define defaults if not already defined via compile flags
#define PWM_MAX 1023
#define PWM_FRQ 1000

#ifndef WSRGB
// Default configuration: BRI_CONTROL + CCT_CONTROL + PWM_CONTROL
#ifndef BRI_CONTROL
#define BRI_CONTROL
#endif
#ifndef CCT_CONTROL
#define CCT_CONTROL
#endif
#ifndef PWM_CONTROL
#define PWM_CONTROL
#endif
#define CCT_START 2700
#define CCT_END 6500
#define CCT_START_MIRED (1000000UL/CCT_START)
#define CCT_END_MIRED (1000000UL/CCT_END)
#endif

#ifdef WSRGB
// WSRGB configuration
#ifndef WS_PIN
#define WS_PIN 14
#endif
#ifndef WS_NUM
#define WS_NUM 8
#endif
#endif

// Global test state
extern std::map<int, int> g_pin_states;
extern std::map<int, int> g_pin_pwm_values;

// Mock implementations of pin functions
inline void test_pinMode(uint8_t pin, uint8_t mode) {
    // Store pin mode if needed
}

inline void test_digitalWrite(uint8_t pin, uint8_t val) {
    g_pin_states[pin] = val;
}

inline int test_digitalRead(uint8_t pin) {
    auto it = g_pin_states.find(pin);
    return (it != g_pin_states.end()) ? it->second : LOW;
}

inline void test_analogWrite(uint8_t pin, int val) {
    g_pin_pwm_values[pin] = val;
}

// Reset test state
inline void reset_test_state() {
    g_pin_states.clear();
    g_pin_pwm_values.clear();
    g_millis = 0;
    g_micros = 0;
}

#endif // TEST_HELPERS_H
