// IR Control Unit Tests
// Tests for IR remote control functionality
// Uses original functions from ir_control.ino via #include

#include <gtest/gtest.h>
#include "test_helpers.h"

// Arduino min/max macros
#ifndef min
#define min(a,b) ((a)<(b)?(a):(b))
#endif
#ifndef max
#define max(a,b) ((a)>(b)?(a):(b))
#endif

// IR configuration for tests
#define IR_PIN D4
#define IR_CODE_TYPE_NEC

// IR button codes in format {address, command}
#define IR_CODE_POWER {0x01, 0x00}
#define IR_CODE_AUTO {0x01, 0x01}
#define IR_CODE_ROUND {0x01, 0x09}
#define IR_CODE_BRI_PLUS {0x01, 0x05}
#define IR_CODE_BRI_MINUS {0x01, 0x04}
#define IR_CODE_BRI_20 {0x01, 0x03}
#define IR_CODE_BRI_50 {0x01, 0x07}
#define IR_CODE_BRI_100 {0x01, 0x0B}
#define IR_CODE_COLOR_TEMP_MINUS {0x01, 0x02}
#define IR_CODE_COLOR_TEMP_PLUS {0x01, 0x0A}
#define IR_CODE_COLOR_TEMP_WARM {0x01, 0x40}
#define IR_CODE_COLOR_TEMP_COLD {0x01, 0x48}
#define IR_CODE_COLOR_TEMP_WHITE {0x01, 0x44}
#define IR_CODE_NIGHT_MODE {0x01, 0x06}

// Define global test variables
size_t PinsCount = 2;
pin_state_str pin_state[2] = {
    {.pin = D6, .state = OFF, .pwm = 0},
    {.pin = D7, .state = OFF, .pwm = 0}
};

main_state_str main_state = {
    .on_procents = 0,
    .led_cct = 0
};

// Mock FileData objects
FileData<FSMock> pinStatesFile(&LittleFS, "/pinStates.dat", 'B', &pin_state, sizeof(pin_state));
FileData<FSMock> mainStateFile(&LittleFS, "/mainState.dat", 'B', &main_state, sizeof(main_state));

WiFiClient espClient;
PubSubClient client(espClient);
WiFiUDP udp;

String device_topic = "homeassistant/light/lamp_test";
String device_sensor_topic = "homeassistant/sensor/lamp_test";
String cmd_topic = "homeassistant/light/lamp_test/cmd";
String state_topic = "homeassistant/light/lamp_test/state";

const char* dev_name = "test_device";
const char* endpoint_type = "lamp";
const char* topic_prefix = "homeassistant";
const char* device_type = "light";

vendor_info_str vendor_info = {
    .manufacturer = "TEST",
    .model = "TEST_MODEL",
    .hw_version = "1.0",
    .sw_version = "1.0.0",
    .support_url = "http://test.com"
};

sensor_state_str sensors_conf[] = {
    {.default_name = "RSSI", .device_class = "signal_strength", .state_class = "measurement", 
     .unit_of_measurement = "dBm", .diag = true, .hide = true, .value = NULL, .func = NULL}
};
size_t sensors_count = 1;

volatile bool send_status_flag = false;
volatile bool send_sensor_flag = false;

unsigned long previousMillis = 0;
String clientId = "test_client";
const char* mqtt_user = "test_user";
const char* mqtt_password = "test_password";

// Arduino map function implementation
long map(long x, long in_min, long in_max, long out_min, long out_max) {
    return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

// Mock debug_printf
size_t debug_printf(const char *format, ...) { return 0; }

// Mock get_sta_mac function
String get_sta_mac() { return "083A8DD1BE93"; }

// Mock process_update function
void process_update() {}

// Mock send functions
void send_power() {}
void send_power(bool power_state) {}
void send_prc(unsigned on_procents) {}
void send_cct(unsigned led_cct) {}
void send_io() {}

// ============ MAIN LOGIC FUNCTIONS (from main_logic.ino) ============

bool set_out(size_t out, bool state) {
    if (out >= PinsCount) return false;
    pin_state[out].state = state;
    return true;
}

bool set_out_pwm(size_t out, unsigned state) {
    if (out >= PinsCount) return false;
    pin_state[out].pwm = state;
    pin_state[out].state = (state > 0) ? ON : OFF;
    return true;
}

bool get_power() {
    for (size_t i = 0; i < PinsCount; i++) {
        if (pin_state[i].state == ON || pin_state[i].pwm > 0) {
            return true;
        }
    }
    return false;
}

void set_power(bool power_state) {
    if (power_state) {
        for (size_t i = 0; i < PinsCount; i++) {
            if (main_state.on_procents > 0) {
                unsigned pwm_val = map(main_state.on_procents, 0, 100, 0, PWM_MAX);
                set_out_pwm(i, pwm_val > 0 ? pwm_val : main_state.on_procents);
            } else {
                set_out_pwm(i, PWM_MAX);
            }
        }
    } else {
        for (size_t i = 0; i < PinsCount; i++) {
            set_out_pwm(i, 0);
        }
    }
}

void set_out_prc(int in) {
    if (in < 0) in = 0;
    if (in > 100) in = 100;
    main_state.on_procents = map(in, 0, 100, 0, PWM_MAX);
    for (size_t i = 0; i < PinsCount; i++) {
        set_out_pwm(i, main_state.on_procents);
    }
}

bool set_out_cct(unsigned in) {
    if (in < CCT_START || in > CCT_END) return false;
    main_state.led_cct = map(in, CCT_START, CCT_END, 0, PWM_MAX);
    return true;
}

// ============ INCLUDE ORIGINAL IR CONTROL CODE ============
// This includes the actual ir_control.ino code to test original functions
#include "../lamp_ctrl2_esp8266/ir_control.ino"

// ============ TEST FIXTURE ============

class IRControlTest : public ::testing::Test {
protected:
    void SetUp() override {
        reset_test_state();
        for (size_t i = 0; i < PinsCount; i++) {
            pin_state[i].state = OFF;
            pin_state[i].pwm = 0;
        }
        main_state.on_procents = 0;
        main_state.led_cct = PWM_MAX / 2; // Mid CCT
    }

    void TearDown() override {}
};

// ============ IR MATCH CODE TESTS ============

TEST_F(IRControlTest, IrMatchCodeExactMatch) {
    EXPECT_TRUE(ir_match_code(0x01, 0x00, 0x01, 0x00));
}

TEST_F(IRControlTest, IrMatchCodeWrongAddress) {
    EXPECT_FALSE(ir_match_code(0x02, 0x00, 0x01, 0x00));
}

TEST_F(IRControlTest, IrMatchCodeWrongCommand) {
    EXPECT_FALSE(ir_match_code(0x01, 0x05, 0x01, 0x00));
}

TEST_F(IRControlTest, IrMatchCodeLower8BitsOnly) {
    // 0x0101 should match 0x01 (lower 8 bits)
    EXPECT_TRUE(ir_match_code(0x0101, 0x0100, 0x01, 0x00));
}

// ============ POWER CONTROL TESTS ============

TEST_F(IRControlTest, IrPowerToggleOn) {
    EXPECT_FALSE(get_power());
    ir_process_command(0x01, 0x00);
    EXPECT_TRUE(get_power());
}

TEST_F(IRControlTest, IrPowerToggleOff) {
    main_state.on_procents = 512;
    set_power(true);
    EXPECT_TRUE(get_power());
    ir_process_command(0x01, 0x00);
    EXPECT_FALSE(get_power());
}

// ============ BRIGHTNESS CONTROL TESTS ============

TEST_F(IRControlTest, IrBrightnessPlus) {
    main_state.on_procents = map(50, 0, 100, 0, PWM_MAX);
    set_power(true);
    unsigned prev_procents = main_state.on_procents;
    ir_process_command(0x01, 0x05);
    // Check that brightness increased
    EXPECT_GT(main_state.on_procents, prev_procents);
    // Check approximate value (allow integer rounding)
    int actual_prc = map(main_state.on_procents, 0, PWM_MAX, 0, 100);
    EXPECT_GE(actual_prc, 58);
    EXPECT_LE(actual_prc, 62);
}

TEST_F(IRControlTest, IrBrightnessMinus) {
    main_state.on_procents = map(50, 0, 100, 0, PWM_MAX);
    set_power(true);
    unsigned prev_procents = main_state.on_procents;
    ir_process_command(0x01, 0x04);
    // Check that brightness decreased
    EXPECT_LT(main_state.on_procents, prev_procents);
    // Check approximate value (allow integer rounding)
    int actual_prc = map(main_state.on_procents, 0, PWM_MAX, 0, 100);
    EXPECT_GE(actual_prc, 38);
    EXPECT_LE(actual_prc, 42);
}

TEST_F(IRControlTest, IrBrightnessMinusFloor) {
    main_state.on_procents = map(5, 0, 100, 0, PWM_MAX);
    set_power(true);
    ir_process_command(0x01, 0x04);
    unsigned expected_on_procents = map(1, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_on_procents, main_state.on_procents);
}

TEST_F(IRControlTest, IrBrightnessPlusCeiling) {
    main_state.on_procents = map(95, 0, 100, 0, PWM_MAX);
    set_power(true);
    ir_process_command(0x01, 0x05);
    EXPECT_EQ(PWM_MAX, main_state.on_procents);
}

TEST_F(IRControlTest, IrBrightness20Preset) {
    ir_process_command(0x01, 0x03);
    EXPECT_TRUE(get_power());
    unsigned expected_on_procents = map(20, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_on_procents, main_state.on_procents);
}

TEST_F(IRControlTest, IrBrightness50Preset) {
    ir_process_command(0x01, 0x07);
    EXPECT_TRUE(get_power());
    unsigned expected_on_procents = map(50, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_on_procents, main_state.on_procents);
}

TEST_F(IRControlTest, IrBrightness100Preset) {
    ir_process_command(0x01, 0x0B);
    EXPECT_TRUE(get_power());
    EXPECT_EQ(PWM_MAX, main_state.on_procents);
}

TEST_F(IRControlTest, IrNightMode) {
    ir_process_command(0x01, 0x06);
    EXPECT_TRUE(get_power());
    unsigned expected_on_procents = map(1, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_on_procents, main_state.on_procents);
}

// ============ CCT CONTROL TESTS ============

TEST_F(IRControlTest, IrColorTempWarm) {
    ir_process_command(0x01, 0x40);
    unsigned expected_led_cct = map(CCT_START, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_led_cct, main_state.led_cct);
}

TEST_F(IRControlTest, IrColorTempCold) {
    ir_process_command(0x01, 0x48);
    unsigned expected_led_cct = map(CCT_END, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_led_cct, main_state.led_cct);
}

TEST_F(IRControlTest, IrColorTempWhite) {
    ir_process_command(0x01, 0x44);
    unsigned long mid_cct = (CCT_START + CCT_END) / 2;
    unsigned expected_led_cct = map(mid_cct, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_led_cct, main_state.led_cct);
}

TEST_F(IRControlTest, IrColorTempPlus) {
    unsigned long start_cct = (CCT_START + CCT_END) / 2;
    main_state.led_cct = map(start_cct, CCT_START, CCT_END, 0, PWM_MAX);
    ir_process_command(0x01, 0x0A);
    // Check that CCT increased by approximately IR_CCT_STEP
    unsigned long actual_cct = map(main_state.led_cct, 0, PWM_MAX, CCT_START, CCT_END);
    unsigned long expected_cct = start_cct + IR_CCT_STEP;
    // Allow small deviation due to integer rounding
    EXPECT_GE(actual_cct, expected_cct - 10);
    EXPECT_LE(actual_cct, expected_cct + 10);
}

TEST_F(IRControlTest, IrColorTempMinus) {
    unsigned long start_cct = (CCT_START + CCT_END) / 2;
    main_state.led_cct = map(start_cct, CCT_START, CCT_END, 0, PWM_MAX);
    ir_process_command(0x01, 0x02);
    unsigned long expected_cct = start_cct - IR_CCT_STEP;
    unsigned expected_led_cct = map(expected_cct, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_led_cct, main_state.led_cct);
}

// ============ AUTO MODE TESTS ============

TEST_F(IRControlTest, IrAutoModeFromOff) {
    EXPECT_FALSE(get_power());
    ir_process_command(0x01, 0x01);
    EXPECT_TRUE(get_power());
    unsigned expected_on_procents = map(20, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_on_procents, main_state.on_procents);
}

TEST_F(IRControlTest, IrAutoModeCycle20To50) {
    main_state.on_procents = map(20, 0, 100, 0, PWM_MAX);
    set_power(true);
    ir_process_command(0x01, 0x01);
    unsigned expected_on_procents = map(50, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_on_procents, main_state.on_procents);
}

TEST_F(IRControlTest, IrAutoModeCycle50To100) {
    main_state.on_procents = map(50, 0, 100, 0, PWM_MAX);
    set_power(true);
    ir_process_command(0x01, 0x01);
    EXPECT_EQ(PWM_MAX, main_state.on_procents);
}

TEST_F(IRControlTest, IrAutoModeCycle100To20) {
    main_state.on_procents = PWM_MAX;
    set_power(true);
    ir_process_command(0x01, 0x01);
    unsigned expected_on_procents = map(20, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_on_procents, main_state.on_procents);
}

// ============ ROUND/DEMO MODE TEST ============

TEST_F(IRControlTest, IrRoundModeTogglesPower) {
    EXPECT_FALSE(get_power());
    ir_process_command(0x01, 0x09);
    EXPECT_TRUE(get_power());
}

// ============ UNKNOWN CODE TEST ============

TEST_F(IRControlTest, IrUnknownCodeNoChange) {
    main_state.on_procents = 512;
    main_state.led_cct = 256;
    unsigned initial_procents = main_state.on_procents;
    unsigned initial_cct = main_state.led_cct;
    ir_process_command(0xFF, 0xFF);
    EXPECT_EQ(initial_procents, main_state.on_procents);
    EXPECT_EQ(initial_cct, main_state.led_cct);
}

// ============ BRIGHTNESS TURNS ON POWER TESTS ============

TEST_F(IRControlTest, IrBrightnessPlusTurnsOnPower) {
    main_state.on_procents = map(50, 0, 100, 0, PWM_MAX);
    set_power(false);
    EXPECT_FALSE(get_power());
    ir_process_command(0x01, 0x05);
    EXPECT_TRUE(get_power());
}

// ============ REPEAT SUPPORT TESTS ============

TEST_F(IRControlTest, IrRepeatSupportedForBrightnessPlus) {
    EXPECT_TRUE(ir_is_repeat_supported_cmd(0x01, 0x05));
}

TEST_F(IRControlTest, IrRepeatSupportedForBrightnessMinus) {
    EXPECT_TRUE(ir_is_repeat_supported_cmd(0x01, 0x04));
}

TEST_F(IRControlTest, IrRepeatSupportedForCctPlus) {
    EXPECT_TRUE(ir_is_repeat_supported_cmd(0x01, 0x0A));
}

TEST_F(IRControlTest, IrRepeatSupportedForCctMinus) {
    EXPECT_TRUE(ir_is_repeat_supported_cmd(0x01, 0x02));
}

TEST_F(IRControlTest, IrRepeatNotSupportedForPower) {
    EXPECT_FALSE(ir_is_repeat_supported_cmd(0x01, 0x00));
}

TEST_F(IRControlTest, IrRepeatNotSupportedForPresets) {
    // BRI_20
    EXPECT_FALSE(ir_is_repeat_supported_cmd(0x01, 0x03));
    // BRI_50
    EXPECT_FALSE(ir_is_repeat_supported_cmd(0x01, 0x07));
    // BRI_100
    EXPECT_FALSE(ir_is_repeat_supported_cmd(0x01, 0x0B));
    // COLOR_TEMP_WARM
    EXPECT_FALSE(ir_is_repeat_supported_cmd(0x01, 0x40));
    // COLOR_TEMP_COLD
    EXPECT_FALSE(ir_is_repeat_supported_cmd(0x01, 0x48));
}

TEST_F(IRControlTest, IrRepeatNotSupportedForUnknownCode) {
    EXPECT_FALSE(ir_is_repeat_supported_cmd(0xFF, 0xFF));
}
