#include <gtest/gtest.h>
#include "lamp_sources.h"
#include "test_helpers.h"

// Define global test variables needed by source files
// Note: These must NOT be const or they will have internal linkage
size_t PinsCount = 2;
pin_state_str pin_state[2] = {
    {.pin = D6, .state = OFF, .pwm = 0},
    {.pin = D7, .state = OFF, .pwm = 0}
};

main_state_str main_state = {
    .on_procents = 0,
    .led_cct = 0
};

FileData<FSMock> pinStatesFile(&LittleFS, "/pinStates.dat", 'B', &pin_state, sizeof(pin_state));
FileData<FSMock> mainStateFile(&LittleFS, "/mainState.dat", 'B', &main_state, sizeof(main_state));

WiFiClient espClient;
PubSubClient client(espClient);
WiFiUDP udp;

String device_topic = "homeassistant/light/lamp_test";
String device_sensor_topic = "homeassistant/sensor/lamp_test";
String cmd_topic = "homeassistant/light/lamp_test/cmd";
String state_topic = "homeassistant/light/lamp_test/state";

const char* dev_name = "test_device";
const char* endpoint_type = "lamp";
const char* topic_prefix = "homeassistant";
const char* device_type = "light";

vendor_info_str vendor_info = {
    .manufacturer = "TEST",
    .model = "TEST_MODEL",
    .hw_version = "1.0",
    .sw_version = "1.0.0",
    .support_url = "http://test.com"
};

sensor_state_str sensors_conf[] = {
    {.default_name = "RSSI", .device_class = "signal_strength", .state_class = "measurement", 
     .unit_of_measurement = "dBm", .diag = true, .hide = true, .value = NULL, .func = NULL}
};
size_t sensors_count = 1;

volatile bool send_status_flag = false;
volatile bool send_sensor_flag = false;

unsigned long previousMillis = 0;
String clientId = "test_client";
const char* mqtt_user = "test_user";
const char* mqtt_password = "test_password";

// ============ TEST FIXTURE ============

class MainLogicTest : public ::testing::Test {
protected:
    void SetUp() override {
        reset_test_state();
        for (size_t i = 0; i < PinsCount; i++) {
            pin_state[i].state = OFF;
            pin_state[i].pwm = 0;
        }
        main_state.on_procents = 0;
        main_state.led_cct = 0;
    }

    void TearDown() override {
    }
};

// ============ TESTS ============

TEST_F(MainLogicTest, SetOutValidPin) {
    bool result = set_out(0, ON);
    EXPECT_TRUE(result);
    EXPECT_EQ(ON, pin_state[0].state);
}

TEST_F(MainLogicTest, SetOutInvalidPin) {
    bool result = set_out(10, ON);
    EXPECT_FALSE(result);
}

TEST_F(MainLogicTest, SetOutPwmValid) {
    bool result = set_out_pwm(0, 512);
    EXPECT_TRUE(result);
    EXPECT_EQ(512, pin_state[0].pwm);
    EXPECT_EQ(ON, pin_state[0].state);
}

TEST_F(MainLogicTest, SetOutPwmZero) {
    pin_state[0].pwm = 100;
    pin_state[0].state = ON;
    
    bool result = set_out_pwm(0, 0);
    EXPECT_TRUE(result);
    EXPECT_EQ(0, pin_state[0].pwm);
    EXPECT_EQ(OFF, pin_state[0].state);
}

TEST_F(MainLogicTest, SetOutPwmInvalidPin) {
    bool result = set_out_pwm(10, 512);
    EXPECT_FALSE(result);
}

TEST_F(MainLogicTest, GetOnceStateAllOff) {
    pin_state[0].state = OFF;
    pin_state[1].state = OFF;
    EXPECT_FALSE(get_once_state());
}

TEST_F(MainLogicTest, GetOnceStateOneOn) {
    pin_state[0].state = ON;
    pin_state[1].state = OFF;
    EXPECT_TRUE(get_once_state());
}

TEST_F(MainLogicTest, SetAllOutOn) {
    set_all_out(ON);
    EXPECT_EQ(ON, pin_state[0].state);
    EXPECT_EQ(ON, pin_state[1].state);
}

TEST_F(MainLogicTest, SetAllOutOff) {
    pin_state[0].state = ON;
    pin_state[1].state = ON;
    
    set_all_out(OFF);
    EXPECT_EQ(OFF, pin_state[0].state);
    EXPECT_EQ(OFF, pin_state[1].state);
}

TEST_F(MainLogicTest, GetPowerOff) {
    pin_state[0].pwm = 0;
    pin_state[0].state = OFF;
    pin_state[1].pwm = 0;
    pin_state[1].state = OFF;
    
    EXPECT_FALSE(get_power());
}

TEST_F(MainLogicTest, GetPowerOn) {
    pin_state[0].pwm = 512;
    pin_state[0].state = ON;
    
    EXPECT_TRUE(get_power());
}

TEST_F(MainLogicTest, SetOutPrc50Percent) {
    main_state.led_cct = PWM_MAX / 2;
    set_out_prc(50);
    
    unsigned expected_procents = map(50, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_procents, main_state.on_procents);
}

TEST_F(MainLogicTest, SetOutPrc100Percent) {
    main_state.led_cct = PWM_MAX / 2;
    set_out_prc(100);
    
    EXPECT_EQ(PWM_MAX, main_state.on_procents);
}

TEST_F(MainLogicTest, SetOutCctValid) {
    unsigned test_cct = 4000;
    bool result = set_out_cct(test_cct);
    
    EXPECT_TRUE(result);
    unsigned expected_led_cct = map(test_cct, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_led_cct, main_state.led_cct);
}

TEST_F(MainLogicTest, SetOutCctInvalidLow) {
    unsigned test_cct = CCT_START - 1;
    bool result = set_out_cct(test_cct);
    
    EXPECT_FALSE(result);
}

TEST_F(MainLogicTest, SetOutCctInvalidHigh) {
    unsigned test_cct = CCT_END + 1;
    bool result = set_out_cct(test_cct);
    
    EXPECT_FALSE(result);
}

TEST_F(MainLogicTest, SetPowerOn) {
    main_state.on_procents = 512;
    
    set_power(true);
    EXPECT_TRUE(get_power());
}

TEST_F(MainLogicTest, SetPowerOff) {
    pin_state[0].state = ON;
    pin_state[1].state = ON;
    
    set_power(false);
    EXPECT_FALSE(get_power());
    EXPECT_EQ(OFF, pin_state[0].state);
    EXPECT_EQ(OFF, pin_state[1].state);
}

TEST_F(MainLogicTest, SwitchPowerToggle) {
    pin_state[0].state = OFF;
    pin_state[1].state = OFF;
    pin_state[0].pwm = 0;
    pin_state[1].pwm = 0;
    main_state.on_procents = 512;
    
    bool prev_state = switch_power();
    EXPECT_FALSE(prev_state);
    EXPECT_TRUE(get_power());
    
    prev_state = switch_power();
    EXPECT_TRUE(prev_state);
    EXPECT_FALSE(get_power());
}
