#include <gtest/gtest.h>
#include "test_helpers.h"

// Test configuration
const char* endpoint_type = "lamp";
const char* topic_prefix = "homeassistant";
const char* dev_name = "test_device";
const char* device_type = "light";

vendor_info_str vendor_info = {
    .manufacturer = "TEST",
    .model = "TEST_MODEL",
    .hw_version = "1.0",
    .sw_version = "1.0.0",
    .support_url = "http://test.com"
};

size_t PinsCount = 2;
pin_state_str pin_state[2] = {
    {.pin = D6, .state = OFF, .pwm = 0},
    {.pin = D7, .state = OFF, .pwm = 0}
};

main_state_str main_state = {
    .on_procents = 0,
    .led_cct = 0
};

// Mock globals
WiFiClient espClient;
PubSubClient client(espClient);
String cmd_topic = "homeassistant/light/lamp_AABBCCDDEEFF/cmd";
String state_topic = "homeassistant/light/lamp_AABBCCDDEEFF/state";
String device_topic = "homeassistant/light/lamp_test";
String device_sensor_topic = "homeassistant/sensor/lamp_test";

sensor_state_str sensors_conf[] = {
    {.default_name = "RSSI", .device_class = "signal_strength", .state_class = "measurement", 
     .unit_of_measurement = "dBm", .diag = true, .hide = true, .value = NULL, .func = NULL}
};
size_t sensors_count = 1;

// Additional globals needed by lamp_sources
unsigned long previousMillis = 0;
String clientId = "test_client";
const char* mqtt_user = "test_user";
const char* mqtt_password = "test_password";
WiFiUDP udp;
FileData<FSMock> pinStatesFile(&LittleFS, "/pinStates.dat", 'B', &pin_state, sizeof(pin_state));
FileData<FSMock> mainStateFile(&LittleFS, "/mainState.dat", 'B', &main_state, sizeof(main_state));
volatile bool send_status_flag = false;
volatile bool send_sensor_flag = false;

// Helper for map - declare before use
long map(long x, long in_min, long in_max, long out_min, long out_max);

// Mock functions called by handlers
bool config_sent = false;
bool sensor_config_sent = false;
bool state_sent = false;

void send_config() {
    config_sent = true;
}

void send_sensor_config() {
    sensor_config_sent = true;
}

void send_all_state() {
    state_sent = true;
}

// Helper for map
long map(long x, long in_min, long in_max, long out_min, long out_max) {
    return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

// Functions from main_logic needed by MQTT handlers
bool set_out(size_t out, bool state) {
    if (out >= PinsCount) return false;
    pin_state[out].state = state;
    return true;
}

bool set_out_pwm(size_t out, unsigned state) {
    if (out >= PinsCount) return false;
    pin_state[out].pwm = state;
    if (state > 0)
        pin_state[out].state = ON;
    else
        pin_state[out].state = OFF;
    return true;
}

void set_all_out(bool state) {
    for (size_t i = 0; i < PinsCount; i++) {
        set_out(i, state);
    }
}

void set_out_prc(int in) {
    bool restore_state = false;
    // Values outside 0-100 are used to restore brightness without changing percentage
    if (in < 0 || in > 100)
        restore_state = true;
    
    int cw_pwm = 0, ww_pwm = 0;
    if (!restore_state) {
        main_state.on_procents = map(in, 0, 100, 0, PWM_MAX);
    }
    cw_pwm = main_state.led_cct;
    ww_pwm = PWM_MAX - main_state.led_cct;
    
    ww_pwm = ((float)ww_pwm / (float)PWM_MAX * (float)main_state.on_procents);
    cw_pwm = ((float)cw_pwm / (float)PWM_MAX * (float)main_state.on_procents);
    set_out_pwm(0, ww_pwm);
    set_out_pwm(1, cw_pwm);
}

bool set_out_cct(unsigned in) {
    if (in < CCT_START || in > CCT_END) return false;
    main_state.led_cct = map(in, CCT_START, CCT_END, 0, PWM_MAX);
    // Value > 100 is used to restore brightness without changing percentage
    set_out_prc(200); // Restore brightness with current percentage
    return true;
}

void set_power(bool power_state) {
    if (!power_state)
        set_all_out(OFF);
    else {
        if (main_state.on_procents > 0)
            set_out_prc(200); // Restore with current brightness
        else
            set_out_prc(100); // Set to 100% if no previous brightness
    }
}

void switch_io(int input, int out) {
    if (out < 0 || out >= (int)PinsCount) return;
    if (input == 1) {
        set_out(out, ON);
    } else if (input == 0) {
        set_out(out, OFF);
    }
}

// MQTT Handler functions to test (from mqtt_utils.ino)
void handle_status_message(const String& payloadStr) {
    if (payloadStr == "online") {
        send_config();
        send_sensor_config();
        send_all_state();
    }
}

void handle_io_command(const String& endTopic, int i_payload) {
    if (endTopic.length() > 0) {
        switch_io(i_payload, endTopic.toInt());
    }
}

void handle_cct_control(int i_payload) {
    if (i_payload >= CCT_START && i_payload <= CCT_END) {
        set_out_cct(i_payload);
    }
}

void handle_cct_mired(int i_payload) {
    if (i_payload >= CCT_END_MIRED && i_payload <= CCT_START_MIRED) {
        unsigned long cct_new = (i_payload == CCT_START_MIRED) ? CCT_START :
                                (i_payload == CCT_END_MIRED) ? CCT_END :
                                1000000UL / i_payload;
        set_out_cct(cct_new);
    }
}

void handle_brightness_control(int i_payload) {
    if (i_payload >= 0 && i_payload <= 100) {
        set_out_prc(i_payload);
    }
}

void handle_power_command(int i_payload) {
    set_power(!!i_payload);
}

// ============ TESTS ============

class MqttTest : public ::testing::Test {
protected:
    void SetUp() override {
        reset_test_state();
        
        // Reset state
        for (size_t i = 0; i < PinsCount; i++) {
            pin_state[i].state = OFF;
            pin_state[i].pwm = 0;
        }
        main_state.on_procents = 0;
        main_state.led_cct = 0;
        
        // Reset mock flags
        config_sent = false;
        sensor_config_sent = false;
        state_sent = false;
    }

    void TearDown() override {
        // Clean up
    }
};

TEST_F(MqttTest, HandleStatusMessageOnline) {
    handle_status_message("online");
    
    EXPECT_TRUE(config_sent);
    EXPECT_TRUE(sensor_config_sent);
    EXPECT_TRUE(state_sent);
}

TEST_F(MqttTest, HandleStatusMessageOther) {
    handle_status_message("offline");
    
    EXPECT_FALSE(config_sent);
    EXPECT_FALSE(sensor_config_sent);
    EXPECT_FALSE(state_sent);
}

TEST_F(MqttTest, HandleIoCommandTurnOn) {
    handle_io_command("0", 1);
    
    EXPECT_EQ(ON, pin_state[0].state);
}

TEST_F(MqttTest, HandleIoCommandTurnOff) {
    pin_state[1].state = ON;
    handle_io_command("1", 0);
    
    EXPECT_EQ(OFF, pin_state[1].state);
}

TEST_F(MqttTest, HandleIoCommandInvalidPin) {
    handle_io_command("99", 1);
    // Should not crash
    EXPECT_TRUE(true);
}

TEST_F(MqttTest, HandleCctControlValid) {
    handle_cct_control(4000);
    
    unsigned expected_cct = map(4000, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_cct, main_state.led_cct);
}

TEST_F(MqttTest, HandleCctControlInvalidLow) {
    unsigned old_cct = main_state.led_cct;
    handle_cct_control(2000);
    
    // Should not change CCT
    EXPECT_EQ(old_cct, main_state.led_cct);
}

TEST_F(MqttTest, HandleCctControlInvalidHigh) {
    unsigned old_cct = main_state.led_cct;
    handle_cct_control(7000);
    
    // Should not change CCT
    EXPECT_EQ(old_cct, main_state.led_cct);
}

TEST_F(MqttTest, HandleCctMiredValid) {
    // 250 mired = 4000K
    handle_cct_mired(250);
    
    // Should have set CCT to 4000K
    unsigned expected_cct = map(4000, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_cct, main_state.led_cct);
}

TEST_F(MqttTest, HandleCctMiredBoundaryStart) {
    handle_cct_mired(CCT_START_MIRED);
    
    unsigned expected_cct = map(CCT_START, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_cct, main_state.led_cct);
}

TEST_F(MqttTest, HandleCctMiredBoundaryEnd) {
    handle_cct_mired(CCT_END_MIRED);
    
    unsigned expected_cct = map(CCT_END, CCT_START, CCT_END, 0, PWM_MAX);
    EXPECT_EQ(expected_cct, main_state.led_cct);
}

TEST_F(MqttTest, HandleBrightnessControl50Percent) {
    handle_brightness_control(50);
    
    unsigned expected_procents = map(50, 0, 100, 0, PWM_MAX);
    EXPECT_EQ(expected_procents, main_state.on_procents);
}

TEST_F(MqttTest, HandleBrightnessControl0Percent) {
    main_state.on_procents = 512;
    handle_brightness_control(0);
    
    EXPECT_EQ(0, main_state.on_procents);
}

TEST_F(MqttTest, HandleBrightnessControl100Percent) {
    handle_brightness_control(100);
    
    EXPECT_EQ(PWM_MAX, main_state.on_procents);
}

TEST_F(MqttTest, HandleBrightnessControlInvalid) {
    unsigned old_procents = main_state.on_procents;
    handle_brightness_control(150);
    
    // Should not change brightness
    EXPECT_EQ(old_procents, main_state.on_procents);
}

TEST_F(MqttTest, HandlePowerCommandOn) {
    main_state.on_procents = 512;
    handle_power_command(1);
    
    // Should have turned on with previous brightness
    EXPECT_TRUE(pin_state[0].pwm > 0 || pin_state[1].pwm > 0);
}

TEST_F(MqttTest, HandlePowerCommandOff) {
    pin_state[0].pwm = 512;
    pin_state[0].state = ON;
    handle_power_command(0);
    
    EXPECT_EQ(OFF, pin_state[0].state);
    EXPECT_EQ(OFF, pin_state[1].state);
}

TEST_F(MqttTest, HandlePowerCommandToggle) {
    main_state.on_procents = 512;
    
    // Turn on
    handle_power_command(1);
    bool first_on = (pin_state[0].pwm > 0 || pin_state[1].pwm > 0);
    EXPECT_TRUE(first_on);
    
    // Turn off
    handle_power_command(0);
    EXPECT_EQ(OFF, pin_state[0].state);
    EXPECT_EQ(OFF, pin_state[1].state);
}
