#include <gtest/gtest.h>
#include "test_helpers.h"

// Sensor configuration
sensor_state_str sensors_conf[] = {
    {.default_name = "RSSI", .device_class = "signal_strength", .state_class = "measurement", 
     .unit_of_measurement = "dBm", .diag = true, .hide = true, .value = NULL, .func = NULL},
    {.default_name = "free_heap", .device_class = "data_size", .state_class = "measurement", 
     .unit_of_measurement = "B", .diag = true, .hide = true, .value = NULL, .func = NULL},
    {.default_name = "test_sensor", .device_class = "temperature", .state_class = "measurement", 
     .unit_of_measurement = "°C", .diag = false, .hide = false, .value = NULL, .func = NULL}
};

size_t sensors_count = sizeof(sensors_conf) / sizeof(sensors_conf[0]);

// Additional globals needed by lamp_sources
const char* dev_name = "test_device";
const char* endpoint_type = "lamp";
const char* topic_prefix = "homeassistant";
const char* device_type = "light";

vendor_info_str vendor_info = {
    .manufacturer = "TEST",
    .model = "TEST_MODEL",
    .hw_version = "1.0",
    .sw_version = "1.0.0",
    .support_url = "http://test.com"
};

size_t PinsCount = 2;
pin_state_str pin_state[2] = {
    {.pin = D6, .state = OFF, .pwm = 0},
    {.pin = D7, .state = OFF, .pwm = 0}
};

main_state_str main_state = {
    .on_procents = 0,
    .led_cct = 0
};

WiFiClient espClient;
PubSubClient client(espClient);
String cmd_topic = "homeassistant/light/lamp_AABBCCDDEEFF/cmd";
String state_topic = "homeassistant/light/lamp_AABBCCDDEEFF/state";
String device_topic = "homeassistant/light/lamp_test";
String device_sensor_topic = "homeassistant/sensor/lamp_test";

unsigned long previousMillis = 0;
String clientId = "test_client";
const char* mqtt_user = "test_user";
const char* mqtt_password = "test_password";
WiFiUDP udp;
FileData<FSMock> pinStatesFile(&LittleFS, "/pinStates.dat", 'B', &pin_state, sizeof(pin_state));
FileData<FSMock> mainStateFile(&LittleFS, "/mainState.dat", 'B', &main_state, sizeof(main_state));
volatile bool send_status_flag = false;
volatile bool send_sensor_flag = false;

// Functions to test (from sens_logic.ino)
unsigned int get_sensor_index(const char *name) {
    for (unsigned int i = 0; i < sensors_count; i++) {
        if (strcmp(sensors_conf[i].default_name, name) == 0) {
            return i;
        }
    }
    return 0;
}

void init_all_sensors() {
    for (size_t i = 0; i < sensors_count; i++) {
        sensors_conf[i].value = (void*)malloc(sizeof(float));
        *(float*)sensors_conf[i].value = 0.0f;
    }
}

void cleanup_sensors() {
    for (size_t i = 0; i < sensors_count; i++) {
        if (sensors_conf[i].value) {
            free(sensors_conf[i].value);
            sensors_conf[i].value = NULL;
        }
    }
}

// ============ TESTS ============

class SensorTest : public ::testing::Test {
protected:
    void SetUp() override {
        reset_test_state();
        
        // Ensure sensors are clean
        for (size_t i = 0; i < sensors_count; i++) {
            if (sensors_conf[i].value) {
                free(sensors_conf[i].value);
                sensors_conf[i].value = NULL;
            }
        }
    }

    void TearDown() override {
        cleanup_sensors();
    }
};

TEST_F(SensorTest, GetSensorIndexFirst) {
    unsigned int index = get_sensor_index("RSSI");
    EXPECT_EQ(0, index);
}

TEST_F(SensorTest, GetSensorIndexMiddle) {
    unsigned int index = get_sensor_index("free_heap");
    EXPECT_EQ(1, index);
}

TEST_F(SensorTest, GetSensorIndexLast) {
    unsigned int index = get_sensor_index("test_sensor");
    EXPECT_EQ(2, index);
}

TEST_F(SensorTest, GetSensorIndexNotFound) {
    unsigned int index = get_sensor_index("nonexistent_sensor");
    EXPECT_EQ(0, index); // Should return 0 when not found
}

TEST_F(SensorTest, GetSensorIndexEmptyString) {
    unsigned int index = get_sensor_index("");
    EXPECT_EQ(0, index);
}

TEST_F(SensorTest, GetSensorIndexNullProtection) {
    // This tests that we don't crash on edge cases
    unsigned int index = get_sensor_index("RSSI");
    EXPECT_EQ(0, index);
}

TEST_F(SensorTest, InitAllSensors) {
    init_all_sensors();
    
    // Check that all sensors have allocated memory
    for (size_t i = 0; i < sensors_count; i++) {
        EXPECT_NE(nullptr, sensors_conf[i].value);
        EXPECT_FLOAT_EQ(0.0f, *(float*)sensors_conf[i].value);
    }
}

TEST_F(SensorTest, InitAllSensorsValuesZero) {
    init_all_sensors();
    
    // All initial values should be zero
    EXPECT_FLOAT_EQ(0.0f, *(float*)sensors_conf[get_sensor_index("RSSI")].value);
    EXPECT_FLOAT_EQ(0.0f, *(float*)sensors_conf[get_sensor_index("free_heap")].value);
    EXPECT_FLOAT_EQ(0.0f, *(float*)sensors_conf[get_sensor_index("test_sensor")].value);
}

TEST_F(SensorTest, SensorValueModification) {
    init_all_sensors();
    
    // Modify a sensor value
    *(float*)sensors_conf[get_sensor_index("RSSI")].value = -75.5f;
    
    EXPECT_FLOAT_EQ(-75.5f, *(float*)sensors_conf[get_sensor_index("RSSI")].value);
}

TEST_F(SensorTest, SensorValueIndependence) {
    init_all_sensors();
    
    // Modify one sensor
    *(float*)sensors_conf[get_sensor_index("RSSI")].value = -75.0f;
    
    // Others should remain zero
    EXPECT_FLOAT_EQ(0.0f, *(float*)sensors_conf[get_sensor_index("free_heap")].value);
    EXPECT_FLOAT_EQ(0.0f, *(float*)sensors_conf[get_sensor_index("test_sensor")].value);
}

TEST_F(SensorTest, SensorCount) {
    // Verify sensor count is correct
    EXPECT_EQ(3, sensors_count);
}

TEST_F(SensorTest, SensorMetadataRssi) {
    unsigned int idx = get_sensor_index("RSSI");
    
    EXPECT_STREQ("RSSI", sensors_conf[idx].default_name);
    EXPECT_STREQ("signal_strength", sensors_conf[idx].device_class);
    EXPECT_STREQ("measurement", sensors_conf[idx].state_class);
    EXPECT_STREQ("dBm", sensors_conf[idx].unit_of_measurement);
    EXPECT_TRUE(sensors_conf[idx].diag);
    EXPECT_TRUE(sensors_conf[idx].hide);
}

TEST_F(SensorTest, SensorMetadataFreeHeap) {
    unsigned int idx = get_sensor_index("free_heap");
    
    EXPECT_STREQ("free_heap", sensors_conf[idx].default_name);
    EXPECT_STREQ("data_size", sensors_conf[idx].device_class);
    EXPECT_STREQ("measurement", sensors_conf[idx].state_class);
    EXPECT_STREQ("B", sensors_conf[idx].unit_of_measurement);
    EXPECT_TRUE(sensors_conf[idx].diag);
    EXPECT_TRUE(sensors_conf[idx].hide);
}

TEST_F(SensorTest, SensorMetadataTestSensor) {
    unsigned int idx = get_sensor_index("test_sensor");
    
    EXPECT_STREQ("test_sensor", sensors_conf[idx].default_name);
    EXPECT_STREQ("temperature", sensors_conf[idx].device_class);
    EXPECT_STREQ("measurement", sensors_conf[idx].state_class);
    EXPECT_STREQ("°C", sensors_conf[idx].unit_of_measurement);
    EXPECT_FALSE(sensors_conf[idx].diag);
    EXPECT_FALSE(sensors_conf[idx].hide);
}

TEST_F(SensorTest, MultipleInitCalls) {
    // First init
    init_all_sensors();
    float* first_ptr = (float*)sensors_conf[0].value;
    
    // Cleanup
    cleanup_sensors();
    
    // Second init
    init_all_sensors();
    float* second_ptr = (float*)sensors_conf[0].value;
    
    // Both should succeed
    EXPECT_NE(nullptr, first_ptr);
    EXPECT_NE(nullptr, second_ptr);
}
