#include <gtest/gtest.h>
#include "lamp_sources.h"
#include "test_helpers.h"

// Test configuration - global variables needed by lamp_sources
const char* dev_name = "test_device";
const char* endpoint_type = "lamp";
const char* topic_prefix = "homeassistant";
const char* device_type = "light";

vendor_info_str vendor_info = {
    .manufacturer = "TEST_MFG",
    .model = "TEST_MODEL",
    .hw_version = "1.0",
    .sw_version = "1.0.0",
    .support_url = "http://test.com"
};

// Mock globals
WiFiClient espClient;
PubSubClient client(espClient);
String device_topic = "homeassistant/light/lamp_AABBCCDDEEFF";
String device_sensor_topic = "homeassistant/sensor/lamp_AABBCCDDEEFF";
String cmd_topic = "homeassistant/light/lamp_AABBCCDDEEFF/cmd";
String state_topic = "homeassistant/light/lamp_AABBCCDDEEFF/state";

size_t PinsCount = 2;
pin_state_str pin_state[2] = {
    {.pin = D6, .state = OFF, .pwm = 0},
    {.pin = D7, .state = OFF, .pwm = 0}
};

main_state_str main_state = {
    .on_procents = 0,
    .led_cct = 0
};

// Mock sensor config
sensor_state_str sensors_conf[] = {
    {.default_name = "RSSI", .device_class = "signal_strength", .state_class = "measurement", 
     .unit_of_measurement = "dBm", .diag = true, .hide = true, .value = NULL, .func = NULL},
    {.default_name = "free_heap", .device_class = "data_size", .state_class = "measurement", 
     .unit_of_measurement = "B", .diag = true, .hide = true, .value = NULL, .func = NULL}
};
size_t sensors_count = sizeof(sensors_conf) / sizeof(sensors_conf[0]);

// Additional globals needed by lamp_sources
unsigned long previousMillis = 0;
String clientId = "test_client";
const char* mqtt_user = "test_user";
const char* mqtt_password = "test_password";
WiFiUDP udp;
FileData<FSMock> pinStatesFile(&LittleFS, "/pinStates.dat", 'B', &pin_state, sizeof(pin_state));
FileData<FSMock> mainStateFile(&LittleFS, "/mainState.dat", 'B', &main_state, sizeof(main_state));
volatile bool send_status_flag = false;
volatile bool send_sensor_flag = false;

// get_sensor_index is not in lamp_sources (sens_logic.ino not included)
// so we need to define it here
unsigned int get_sensor_index(const char *name) {
    for (unsigned int i = 0; i < sensors_count; i++) {
        if (strcmp(sensors_conf[i].default_name, name) == 0) {
            return i;
        }
    }
    return 0;
}

// ============ TESTS ============

class UtilsTest : public ::testing::Test {
protected:
    void SetUp() override {
        reset_test_state();
        client.setConnected(true);
        
        // Reset pin and main state
        for (size_t i = 0; i < PinsCount; i++) {
            pin_state[i].state = OFF;
            pin_state[i].pwm = 0;
        }
        main_state.on_procents = 0;
        main_state.led_cct = 0;
    }

    void TearDown() override {
        // Clean up
    }
};

TEST_F(UtilsTest, SendStatus) {
    send_status("test", "value");
    
    String expected_topic = state_topic + "/test";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
    EXPECT_STREQ("value", client.getLastPublishPayload().c_str());
}

TEST_F(UtilsTest, SendStatusToMqttConnected) {
    client.setConnected(true);
    send_status_to_mqtt("test", "123");
    
    String expected_topic = state_topic + "/test";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
}

TEST_F(UtilsTest, SendStatusToMqttDisconnected) {
    client.setConnected(false);
    send_status_to_mqtt("test", "123");
    
    // Should not publish when disconnected - no assertion needed, just verify no crash
}

TEST_F(UtilsTest, SendIo) {
    pin_state[0].state = ON;
    pin_state[1].state = OFF;
    
    send_io();
    
    // Should have published state for both pins
    String expected_topic = state_topic + "/lamp/1";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
}

TEST_F(UtilsTest, GetPowerWithPwm) {
    pin_state[0].pwm = 512;
    pin_state[0].state = ON;
    
    EXPECT_TRUE(get_power());
}

TEST_F(UtilsTest, GetPowerNoPwm) {
    pin_state[0].pwm = 0;
    pin_state[0].state = OFF;
    pin_state[1].pwm = 0;
    pin_state[1].state = OFF;
    
    EXPECT_FALSE(get_power());
}

TEST_F(UtilsTest, SendPowerOn) {
    send_power(true);
    
    String expected_topic = state_topic + "/lamp/power";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
    EXPECT_STREQ("1", client.getLastPublishPayload().c_str());
}

TEST_F(UtilsTest, SendPowerOff) {
    send_power(false);
    
    String expected_topic = state_topic + "/lamp/power";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
    EXPECT_STREQ("0", client.getLastPublishPayload().c_str());
}

TEST_F(UtilsTest, SendPrc) {
    send_prc(512); // 50% of PWM_MAX (1023)
    
    String expected_topic = state_topic + "/lamp/prc";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
    
    // 512 mapped from 0-1023 to 0-100 should be about 50
    unsigned expected_prc = map(512, 0, PWM_MAX, 0, 100);
    EXPECT_STREQ(String(expected_prc).c_str(), client.getLastPublishPayload().c_str());
}

TEST_F(UtilsTest, SendCct) {
    send_cct(512); // Mid-range CCT
    
    // Should publish both cct and cct_mired
    String expected_topic = state_topic + "/lamp/cct_mired";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
}

TEST_F(UtilsTest, SendAllState) {
    main_state.on_procents = 512;
    main_state.led_cct = 256;
    pin_state[0].pwm = 512;
    pin_state[0].state = ON;
    
    send_all_state();
    
    // Should have published power, prc, and cct
    // Last publish should be cct_mired
    String expected_topic = state_topic + "/lamp/cct_mired";
    EXPECT_STREQ(expected_topic.c_str(), client.getLastPublishTopic().c_str());
}

TEST_F(UtilsTest, SendDeviceInfo) {
    JsonDocument config;
    send_device_info(config);
    
    // Verify that device info is set (this is a basic test)
    // In a real scenario, we'd check the JSON structure
    EXPECT_TRUE(true); // Just verify it doesn't crash
}

TEST_F(UtilsTest, GetSensorIndexFound) {
    unsigned int index = get_sensor_index("RSSI");
    EXPECT_EQ(0, index);
    
    index = get_sensor_index("free_heap");
    EXPECT_EQ(1, index);
}

TEST_F(UtilsTest, GetSensorIndexNotFound) {
    unsigned int index = get_sensor_index("nonexistent");
    EXPECT_EQ(0, index); // Returns 0 when not found
}
