/**
 * WSRGB send_config() Integration Tests
 * 
 * These tests verify that send_config() function in WSRGB mode generates
 * correct unique_id, name, and topic formats for ALL entity types.
 * 
 * The tests use REAL code from send_config() compiled with WSRGB flag.
 * 
 * Expected format: {prefix}_{MAC}_{index} or {prefix}_{MAC}
 * Example: lamp_083A8DCCC915_0
 * 
 * WRONG format (breaks backward compatibility):
 * {prefix}_relay_{index}_{MAC}
 * Example: lamp_relay_0_083A8DCCC915
 * 
 * These tests should FAIL on the current branch (proving the bug exists)
 * and should PASS after the fix is applied.
 */

#include <gtest/gtest.h>
#include <cstdlib>
#include <ctime>
#include <algorithm>

// Must define WSRGB before including test_helpers
#define WSRGB
#define WS_PIN 14
#define WS_NUM 8

#include "../test_helpers.h"
#include "../lamp_sources.h"

// Global test state
std::map<int, int> g_pin_states;
std::map<int, int> g_pin_pwm_values;
unsigned long g_millis = 0;
unsigned long g_micros = 0;

// Hardware configuration for WSRGB
size_t PinsCount = 2;
pin_state_str pin_state[] = {
    {.pin = 12, .state = ON, .pwm = 0},
    {.pin = 14, .state = OFF, .pwm = 0}
};

main_state_str main_state = {.on_procents = 100, .rgb_r = 255, .rgb_g = 255, .rgb_b = 255};

// Mock file data
FSMock LittleFS;
FileData<FSMock> pinStatesFile(&LittleFS, "/pinStates.dat", 'B', &pin_state, sizeof(pin_state));
FileData<FSMock> mainStateFile(&LittleFS, "/mainState.dat", 'B', &main_state, sizeof(main_state));

// Network
WiFiClient espClient;
PubSubClient client(espClient);
WiFiUDP udp;

// Topics
String device_topic;
String device_sensor_topic;
String cmd_topic;
String state_topic;
String clientId;

// Configuration
const char* dev_name = "rgbl2";
const char* endpoint_type = "lamp";
const char* topic_prefix = "homeassistant";
const char* device_type = "light";
const char* mqtt_user = "test";
const char* mqtt_password = "test";

vendor_info_str vendor_info = {
    .manufacturer = "A1DEV",
    .model = "light control esp8266 rgbl2",
    .hw_version = "0.2",
    .sw_version = "1.3",
    .support_url = "https://support.url"
};

// Minimal sensor config for tests
sensor_state_str sensors_conf[] = {
    {"test_sensor", "temperature", "measurement", "°C", false, false, nullptr, nullptr}
};
size_t sensors_count = 1;

volatile bool send_status_flag = false;
volatile bool send_sensor_flag = false;
unsigned long previousMillis = 0;

// Helper function to generate random MAC address
String generateRandomMAC() {
    char mac[18];
    unsigned int seed = static_cast<unsigned int>(time(nullptr)) + static_cast<unsigned int>(clock());
    srand(seed);
    
    int byte1 = rand() % 256;
    int byte2 = rand() % 256;
    int byte3 = rand() % 256;
    int byte4 = rand() % 256;
    int byte5 = rand() % 256;
    int byte6 = rand() % 256;
    
    sprintf(mac, "%02X:%02X:%02X:%02X:%02X:%02X", byte1, byte2, byte3, byte4, byte5, byte6);
    return String(mac);
}

class WsrgbSendConfigTest : public ::testing::Test {
protected:
    String test_mac;
    
    void SetUp() override {
        g_pin_states.clear();
        g_pin_pwm_values.clear();
        g_millis = 0;
        g_micros = 0;
        
        test_mac = generateRandomMAC();
        WiFi.setMacAddress(test_mac.c_str());
        WiFi.setLocalIP("192.168.4.139");
        
        client.setServer("192.168.4.197", 1883);
        client.setConnected(true);
        client.clearPublished();
        
        String mac = get_sta_mac();
        device_topic = String(topic_prefix) + "/" + String(device_type) + "/" + String(endpoint_type) + "_" + mac;
        cmd_topic = device_topic + "/cmd";
        state_topic = device_topic + "/state";
        device_sensor_topic = String(topic_prefix) + "/sensor/" + String(endpoint_type) + "_" + mac;
    }
    
    void TearDown() override {
        client.clearPublished();
    }
    
    String getFormattedMAC() {
        String result;
        for (unsigned int i = 0; i < test_mac.length(); i++) {
            if (test_mac[i] != ':') {
                result += test_mac[i];
            }
        }
        return result;
    }
    
    // Helper to find char starting from a specific position
    int indexOfFrom(const String& str, char ch, int fromPos) {
        if (fromPos >= (int)str.length()) return -1;
        for (int i = fromPos; i < (int)str.length(); i++) {
            if (str[i] == ch) return i;
        }
        return -1;
    }
    
    // Extract JSON value by key
    String extractJsonValue(const String& payload, const String& key) {
        String searchKey = "\"" + key + "\"";
        int keyPos = payload.indexOf(searchKey);
        if (keyPos < 0) return "";
        
        int colonPos = indexOfFrom(payload, ':', keyPos);
        if (colonPos < 0) return "";
        
        int valueStart = colonPos + 1;
        while (valueStart < (int)payload.length() && (payload[valueStart] == ' ' || payload[valueStart] == '\t')) {
            valueStart++;
        }
        
        if (valueStart >= (int)payload.length()) return "";
        
        char startChar = payload[valueStart];
        if (startChar == '"') {
            int endQuote = indexOfFrom(payload, '"', valueStart + 1);
            if (endQuote < 0) return "";
            return payload.substring(valueStart + 1, endQuote);
        } else {
            int endPos = valueStart;
            while (endPos < (int)payload.length() && 
                   payload[endPos] != ',' && 
                   payload[endPos] != '}' &&
                   payload[endPos] != ']') {
                endPos++;
            }
            String value = payload.substring(valueStart, endPos);
            while (value.length() > 0 && (value[0] == ' ' || value[0] == '\t')) {
                value = value.substring(1);
            }
            while (value.length() > 0 && (value[value.length()-1] == ' ' || value[value.length()-1] == '\t')) {
                value = value.substring(0, value.length()-1);
            }
            return value;
        }
    }
};

// ============================================================================
// Tests for ALL entity types published by send_config()
// These tests verify unique_id format from REAL send_config() output
// ============================================================================

TEST_F(WsrgbSendConfigTest, AllUniqueIdsMustNotContainRelayKeyword) {
    // Call REAL send_config() from utils.ino compiled with WSRGB
    send_config(0);
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    int configCount = 0;
    
    for (const auto& msg : published) {
        // Only check config topics
        if (msg.topic.indexOf("/config") < 0) continue;
        
        configCount++;
        
        String payload = msg.payload;
        String uniqueId = extractJsonValue(payload, "unique_id");
        
        // CRITICAL TEST: unique_id should NOT contain "_relay_"
        EXPECT_EQ(uniqueId.indexOf("_relay_"), -1)
            << "unique_id '" << uniqueId.c_str() 
            << "' should NOT contain '_relay_'. "
            << "Expected format: {endpoint_type}_{MAC} or {endpoint_type}_{MAC}_{index}";
    }
    
    // Should have published at least main config + relay configs
    EXPECT_GE(configCount, 1) << "Should have published at least one config";
}

TEST_F(WsrgbSendConfigTest, AllNamesMustNotContainRelayKeyword) {
    send_config(0);
    
    auto published = client.getPublishedMessages();
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String name = extractJsonValue(payload, "name");
        
        // CRITICAL TEST: name should NOT contain "_relay_"
        EXPECT_EQ(name.indexOf("_relay_"), -1)
            << "name '" << name.c_str() 
            << "' should NOT contain '_relay_'. "
            << "Expected format: {dev_name}_{MAC} or {dev_name}_{MAC}_{index}";
    }
}

TEST_F(WsrgbSendConfigTest, AllConfigTopicsMustNotContainRelayKeyword) {
    send_config(0);
    
    auto published = client.getPublishedMessages();
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        // Config topic should NOT contain "_relay_"
        EXPECT_EQ(msg.topic.indexOf("_relay_"), -1)
            << "Config topic '" << msg.topic.c_str() 
            << "' should NOT contain '_relay_'. "
            << "Expected format: device_topic/config or device_topic_{index}/config";
    }
}

TEST_F(WsrgbSendConfigTest, MacMustComeBeforeIndexInAllUniqueIds) {
    send_config(0);
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String uniqueId = extractJsonValue(payload, "unique_id");
        
        // Check if this is an indexed entity (ending with _N)
        bool isIndexedEntity = false;
        if (uniqueId.length() > 2) {
            char lastChar = uniqueId[uniqueId.length() - 1];
            char secondLastChar = uniqueId[uniqueId.length() - 2];
            if (lastChar >= '0' && lastChar <= '9' && secondLastChar == '_') {
                isIndexedEntity = true;
            }
        }
        
        if (isIndexedEntity) {
            int macPos = uniqueId.indexOf(mac);
            
            // Find the last underscore before the index digit
            int lastUnderscorePos = -1;
            for (int j = uniqueId.length() - 2; j >= 0; j--) {
                if (uniqueId[j] == '_') {
                    lastUnderscorePos = j;
                    break;
                }
            }
            
            // MAC must come BEFORE the index
            if (macPos >= 0 && lastUnderscorePos >= 0) {
                EXPECT_LT(macPos, lastUnderscorePos)
                    << "In unique_id '" << uniqueId.c_str() 
                    << "', MAC (" << mac.c_str() << ") must come BEFORE index.";
            }
        }
    }
}

TEST_F(WsrgbSendConfigTest, MacMustComeBeforeIndexInAllNames) {
    send_config(0);
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String name = extractJsonValue(payload, "name");
        
        // Check if this is an indexed entity
        bool isIndexedEntity = false;
        if (name.length() > 2) {
            char lastChar = name[name.length() - 1];
            char secondLastChar = name[name.length() - 2];
            if (lastChar >= '0' && lastChar <= '9' && secondLastChar == '_') {
                isIndexedEntity = true;
            }
        }
        
        if (isIndexedEntity) {
            int macPos = name.indexOf(mac);
            
            int lastUnderscorePos = -1;
            for (int j = name.length() - 2; j >= 0; j--) {
                if (name[j] == '_') {
                    lastUnderscorePos = j;
                    break;
                }
            }
            
            if (macPos >= 0 && lastUnderscorePos >= 0) {
                EXPECT_LT(macPos, lastUnderscorePos)
                    << "In name '" << name.c_str() 
                    << "', MAC (" << mac.c_str() << ") must come BEFORE index.";
            }
        }
    }
}

TEST_F(WsrgbSendConfigTest, AllUniqueIdsMustContainMac) {
    send_config(0);
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String uniqueId = extractJsonValue(payload, "unique_id");
        
        EXPECT_GE(uniqueId.indexOf(mac), 0)
            << "unique_id '" << uniqueId.c_str() 
            << "' should contain MAC address '" << mac.c_str() << "'";
    }
}

// ============================================================================
// Specific format verification tests
// ============================================================================

TEST_F(WsrgbSendConfigTest, RelayEntityUniqueIdMustMatchMainBranchFormat) {
    send_config(0);
    
    String mac = getFormattedMAC();
    auto published = client.getPublishedMessages();
    
    // Expected format for relay entities (matching main branch):
    // unique_id: lamp_083A8DCCC915_0
    // name: rgbl2_083A8DCCC915_0
    
    // Wrong format (current WSRGB bug):
    // unique_id: lamp_relay_0_083A8DCCC915
    // name: rgbl2_relay_0_083A8DCCC915
    
    for (const auto& msg : published) {
        if (msg.topic.indexOf("/config") < 0) continue;
        
        String payload = msg.payload;
        String uniqueId = extractJsonValue(payload, "unique_id");
        String name = extractJsonValue(payload, "name");
        
        // Skip main RGB config (which doesn't have index)
        if (uniqueId == String(endpoint_type) + "_" + mac) continue;
        
        // For indexed entities, verify correct format
        if (uniqueId.length() > 2) {
            char lastChar = uniqueId[uniqueId.length() - 1];
            if (lastChar >= '0' && lastChar <= '9') {
                // This is an indexed entity
                int index = lastChar - '0';
                
                // Expected unique_id: lamp_MAC_index
                String expectedUniqueId = String(endpoint_type) + "_" + mac + "_" + String(index);
                EXPECT_EQ(uniqueId, expectedUniqueId)
                    << "Indexed entity unique_id should be in format: {endpoint_type}_{MAC}_{index}";
                
                // Expected name: dev_name_MAC_index
                String expectedName = String(dev_name) + "_" + mac + "_" + String(index);
                EXPECT_EQ(name, expectedName)
                    << "Indexed entity name should be in format: {dev_name}_{MAC}_{index}";
            }
        }
    }
}
